/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : kf32a156_pm.c                               	      *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This file provides the power management (PM) related  *
 *  					functions, including:								  *
 *          			+ Power management(PM) control function               *
 *          			+ Power management(PM) status flag management function*
 *          			+ Power management(PM) calibration control function	  *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/

/******************************************************************************
**                      		Include Files                                **
******************************************************************************/


#include "kf32a156_pm.h"
#include "kf32a156_osc.h"


/**
  *   ##### Դ(PM)ƹܺ #####
  */
/**
  *   IO״̬ʹܡ
  *   NewState: IO״̬ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_IO_Latch_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴIOLATCHλ ----------------*/
    if (NewState != FALSE)
    {
        /*  IO״̬ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_IOLATCH_POS);
    }
    else
    {
        /* IO״̬δ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_IOLATCH_POS);
    }
}

/**
  *   ȡIO״̬档
  *   ޡ
  *   1:IO״̬棻0:IO״̬δ档
  */
FlagStatus
PM_Get_IO_Latch_Status (void)
{
    /*---------------- ȡPM_CTL0ĴIOLATCHλ ----------------*/
    if (PM_CTL0 & PM_CTL0_IOLATCH)
    {
        /* IO״̬ */
        return SET;
    }
    else
    {
        /* IO״̬δ */
        return RESET;
    }
}

/**
  *   ڲƵʹܡ
  *   NewState: ڲƵʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Internal_Low_Frequency_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴLSIENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹڲƵ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LSIEN_POS);
    }
    else
    {
        /* δʹڲƵ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LSIEN_POS);
    }
}

/**
  *   ⲿƵʹܡ
  *   NewState: ⲿƵʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_External_Low_Frequency_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴLSEENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹⲿƵ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LSEEN_POS);
    }
    else
    {
        /* δʹⲿƵ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LSEEN_POS);
    }
}

/**
  *   ⲿƵʱʹܡ
  *   NewState: ⲿƵʱʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_External_Low_Frequency_Clock_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴLSEEXENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ֱӴⲿƵʱźΪ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LSEEXEN_POS);
    }
    else
    {
        /* ֹⲿʱ빦 */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LSEEXEN_POS);
    }
}

/**
  *   BGʹܡ
  *   NewState: BGʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Main_Bandgap_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴMRBGENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹBGģ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_MRBGEN_POS);
    }
    else
    {
        /* δʹBGģ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_MRBGEN_POS);
    }
}

/**
  *   LDO18ʹܡ
  *   NewState: LDO18ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_LDO18_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴLDO18ENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹLDO18ģ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LDO18EN_POS);
    }
    else
    {
        /* ֹLDO18ģ飬Power18flash */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LDO18EN_POS);
    }
}

/**
  *   ñĴģλ
  *   BkpReset: Ĵģλ״̬ȡֵΪ:
  *                   PERIPHERAL_RST_STATUS: Ĵģ鴦ڸλ״̬
  *                   PERIPHERAL_OUTRST_STATUS: Ĵģ˳λ
  *   ޡ
  */
void
PM_Backup_Registers_Reset_Config (uint32_t BkpReset)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PERIPHERAL_RST(BkpReset));

    /*---------------- PM_CTL0ĴBKPREGCLRλ ----------------*/
    if (BkpReset != PERIPHERAL_RST_STATUS)
    {
        /* Ĵģ˳λ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_BKPREGCLR_POS);
    }
    else
    {
        /* Ĵģ鴦ڸλ״̬ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_BKPREGCLR_POS);
    }
}

/**
  *   IWDTģλ
  *   IWDTReset: IWDTģλ״̬ȡֵΪ:
  *                   PERIPHERAL_RST_STATUS: IWDTڸλ״̬
  *                   PERIPHERAL_OUTRST_STATUS: IWDT˳λ
  *   ޡ
  */
void
PM_Independent_Watchdog_Reset_Config (uint32_t IWDTReset)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PERIPHERAL_RST(IWDTReset));

    /*---------------- PM_CTL0ĴIWDTCLRλ ----------------*/
    if (IWDTReset != PERIPHERAL_RST_STATUS)
    {
        /* IWDT˳λ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_IWDTCLR_POS);
    }
    else
    {
        /* IWDTڸλ״̬ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_IWDTCLR_POS);
    }
}

/**
  *   DPRAMstandbyģʽ¹á
  *   WorkMode: DPRAM_Astandbyģʽ¹ãȡֵΪ:
  *                   PM_DPRAM_IN_STANDBY_POWER_DOWN: SRAMAstandbyģʽµ
  *                   PM_DPRAM_IN_STANDBY_KEEP_DATA: SRAMAstandbyģʽ±
  *   ޡ
  */
void
PM_DPRAM_In_Standby_Work_Mode_Config (uint32_t WorkMode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_DPRAM_IN_STANDBY(WorkMode));

    /*---------------- PM_CTL0ĴDPRAMASELλ ----------------*/
    if (WorkMode != PM_DPRAM_IN_STANDBY_POWER_DOWN)
    {
        /* DPRAM_Astandbyģʽ± */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_DPRAMASEL_POS);
    }
    else
    {
        /* SRAMAstandbyģʽµ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_DPRAMASEL_POS);
    }
}

/**
  *   LPRAM stop1  standby ģʽ¹á
  *   WorkMode: LPRAM stop1  standby ģʽ¹ãȡֵΪ:
  *                   PM_LPRAM_IN_STANDBY_POWER_DOWN: LPRAMstandbyģʽµ
  *                   PM_LPRAM_IN_STANDBY_KEEP_DATA: LPRAMstandbyģʽ±
  *   ޡ
  */
void
PM_LPRAM_In_Standby_Stop1_Work_Mode_Config (uint32_t WorkMode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_LPSRAM_IN_STANDBY(WorkMode));

    /*---------------- PM_CTL0ĴSRAMASELλ ----------------*/
    if (WorkMode != PM_LPRAM_IN_STANDBY_POWER_DOWN)
    {
        /* LPRAMstop1  standbyģʽ± */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LPRAMSEL_POS);
    }
    else
    {
        /* LPRAMstop1  standbyģʽµ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LPRAMSEL_POS);
    }
}

/**
  *   BKP_PORʱʱ䡣
  *   DelayTime: BKP_PORʱʱ䣬ȡֵΪ:
  *                    DELAY_TIME_2ms: 2ms
  *                    DELAY_TIME_32us: 32us
  *   ޡ
  */
void
PM_Backup_POR_Delay_Time_Config (uint32_t DelayTime)
{
    /* У */
    CHECK_RESTRICTION(CHECK_DELAY_TIME(DelayTime));

    /*---------------- PM_CTL0ĴBKPPORDELAYSELλ ----------------*/
    if (DelayTime != DELAY_TIME_2ms)
    {
        /* ʱ1ms */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_BKPPORDELAYSEL_POS);
    }
    else
    {
        /* ʱ32us */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_BKPPORDELAYSEL_POS);
    }
}

/**
  *   PORPOR12POR18ʱʱ䡣
  *   DelayTime: PORPOR12POR18ʱʱ䣬ȡֵΪ:
  *                    DELAY_TIME_2ms: 2ms
  *                    DELAY_TIME_32us: 32us
  *   ޡ
  */
void
PM_Main_POR_Delay_Time_Config (uint32_t DelayTime)
{
    /* У */
    CHECK_RESTRICTION(CHECK_DELAY_TIME(DelayTime));

    /*---------------- PM_CTL0ĴPORDELAYSELλ ----------------*/
    if (DelayTime != DELAY_TIME_2ms)
    {
        /* ʱ2ms */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_PORDELAYSEL_POS);
    }
    else
    {
        /* ʱ32us */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_PORDELAYSEL_POS);
    }
}

/**
  *   õ͹IOڡ
  *   PeripheralPort: ͹IOڣȡֵΪ:
  *                         PM_GENERAL_PURPOSE_IO_PORT: ͨIO
  *                         PM_LOW_POWER_IO_PORT: ͹רIO
  *   ޡ
  */
void
PM_Peripheral_IO_Port_Config (uint32_t PeripheralPort)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_LOW_POWER_IO(PeripheralPort));

    /*---------------- PM_CTL0ĴPHERIIOSELλ ----------------*/
    if (PeripheralPort != PM_GENERAL_PURPOSE_IO_PORT)
    {
        /* ͹רIO */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_PHERIIOSEL_POS);
    }
    else
    {
        /* ͨIO */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_PHERIIOSEL_POS);
    }
}

/**
  *   ͹ģʽڲƵУ׼ֵ
  *   NewState: HSIڲƵУ׼ֵλ,ȡֵΪTRUE  FALSE
  *   
  */
void
PM_OCAL0LOCK_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴOCALLOCKλ ----------------*/
    if (NewState != FALSE)
    {
        /* 澧У׼ֵ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_OCALLOCK_POS);
    }
    else
    {
        /* 澧У׼ֵ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_OCALLOCK_POS);
    }
}

/**
  *   ͹ģʽMEMMݱʹ
  *   NewState: MEMMݱʹλ,ȡֵΪTRUE  FALSE
  *   
  */
void
PM_MEMSEL_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴOCALLOCKλ ----------------*/
    if (NewState != FALSE)
    {
        /* 澧У׼ֵ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_MEMSEL_POS);
    }
    else
    {
        /* 澧У׼ֵ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_MEMSEL_POS);
    }
}

/**
  *   FLASHԴضʹܡ
  *   NewState: FLASHԴضʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Flash_Power_Off_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL2ĴVF12ENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ضFLASHԴأFLASHģ */
        SFR_CLR_BIT_ASM(PM_CTL2, PM_CTL2_VF12EN_POS);
    }
    else
    {
        /* δضFLASHԴ */
        SFR_SET_BIT_ASM(PM_CTL2, PM_CTL2_VF12EN_POS);
    }
}

/**
 *  @brief : Config the CCP0 CLk is INTLF
 *  @param in :NewState TRUE OR FALSE
 *  @param out :None
 *  @retval :None
 */
void PM_CCP0LPEN_ENABLE(FunctionalState NewState)
{
	/* param check */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));
	if (NewState != FALSE)
    {
		/* Set the INTLF as the CCP0 clk */
        SFR_SET_BIT_ASM(PM_CTL2, PM_CTL2_CCP0LPEN_POS);
    }
    else
    {
        /* Clear the INTLF as the CCP0 clk */
        SFR_CLR_BIT_ASM(PM_CTL2, PM_CTL2_CCP0LPEN_POS);
    }
}

/**
  *   CCP0ڲƵʱʹܡ
  *   NewState: CCP0ڲƵʱʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_CCP0CLKLPEN_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL2ĴCCP0CLKLPENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ڲƵʱΪ CCP0 ʱ */
        SFR_SET_BIT_ASM(PM_CTL2, PM_CTL2_CCP0CLKLPEN_POS);
    }
    else
    {
        /* ֹڲƵʱΪ CCP0 ʱ */
        SFR_CLR_BIT_ASM(PM_CTL2, PM_CTL2_CCP0CLKLPEN_POS);
    }
}

/**
  *   ñдʹܡ
  *   NewState: дʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Backup_Write_And_Read_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴBKPWRλ ----------------*/
    if (NewState != FALSE)
    {
        /* CPUԱڼĴжд */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_BKPWR_POS);
    }
    else
    {
        /* ֹCPUԱڼĴжд */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_BKPWR_POS);
    }
}

/**
  *   òοѹʹܡ
  *   NewState: οѹʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_VREF_Software_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴVREFENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܲοѹ */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
    else
    {
        /* δʹܲοѹ*/
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
}

/**
  *   òοѹλѡ
  *   NewState: οѹλѡ
  * 			PM_VREF_VOLTAGE_4P0V
  * 			PM_VREF_VOLTAGE_1P5V
  * 			PM_VREF_VOLTAGE_2P0V
  * 			PM_VREF_VOLTAGE_3P0V
  *   ޡ
  */
void
PM_VREF_SELECT (uint32_t Voltage)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_VREF_VOLTAGE(Voltage));

    /*---------------- PM_CTL0ĴVREFSELλ ----------------*/
    PM_CTL1 = SFR_Config (PM_CTL1, ~PM_CTL1_VREFSEL, Voltage);
}

/**
  *   õ͹ĵѹLPRʹܡ
  *   NewState: ͹ĵѹLPRʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_LPR_Software_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴLPRENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹLPR */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LPREN_POS);
    }
    else
    {
        /* δʹLPR */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LPREN_POS);
    }
}

/**
  *   õ͹ģʽ
  *   LowPowerMode: ͹ģʽȡֵΪ:
  *                       PM_LOW_POWER_MODE_STOP_0: ֹͣģʽ0
  *                       PM_LOW_POWER_MODE_STOP_1: ֹͣģʽ1
  *                       PM_LOW_POWER_MODE_STANDBY: ģʽ
  *                       PM_LOW_POWER_MODE_SHUTDOWN: ضģʽ
  *   ޡ
  */
void
PM_Low_Power_Mode_Config (uint32_t LowPowerMode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_LOW_POWER_MODE(LowPowerMode));

    /*---------------- PM_CTL0ĴLPMSλ ----------------*/
    PM_CTL0 = SFR_Config (PM_CTL0, ~PM_CTL0_LPMS, LowPowerMode);
}

/**
  *   BORʹܡ
  *   NewState: BORʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_BOR_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴBORENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹBOR */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_BOREN_POS);
    }
    else
    {
        /* ֹBOR */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_BOREN_POS);
    }
}

/**
  *   LPBORʹܡ
  *   NewState: LPBORʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Low_Power_BOR_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴLPBORENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹLPBOR */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_LPBOREN_POS);
    }
    else
    {
        /* ֹLPBOR */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_LPBOREN_POS);
    }
}

/**
  *   TEMPSENSORʹܡ
  *   NewState: TEMPSENSORʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Temperature_Sensor_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴTSENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹTEMPSENSOR */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_TSEN_POS);
    }
    else
    {
        /* ֹTEMPSENSOR */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_TSEN_POS);
    }
}

/**
  *   TEMPSENSORBUFFERʹܡ
  *   NewState: TEMPSENSORBUFFERʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Temperature_Sensor_Buffer_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴTSOEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹTEMPSENSORBUFFER */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_TSOE_POS);
    }
    else
    {
        /* ֹTEMPSENSORBUFFER */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_TSOE_POS);
    }
}

/**
  *   òοѹ2Vģʹܡ
  *   NewState: οѹ2Vģʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Reference_Voltage_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴVREFENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܲοѹ2Vģ */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
    else
    {
        /* ֹοѹ2Vģ */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
}

/**
  *   ڲBUFFERʱʹܡ
  *   NewState: ڲBUFFERʱʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Internal_Test_Buffer_Clock_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴBUFCLKENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹڲBUFFERʱ */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_BUFCLKEN_POS);
    }
    else
    {
        /* ֹڲBUFFERʱ */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_BUFCLKEN_POS);
    }
}

/**
  *   ڲBUFFERʱӷƵ
  *   SclkScaler: ڲBUFFERʱӷƵȡֵΪ:
  *                     PM_BUFFER_SCLK_DIV_1: SCLK
  *                     PM_BUFFER_SCLK_DIV_2: SCLK/2
  *                     PM_BUFFER_SCLK_DIV_4: SCLK/4
  *                     PM_BUFFER_SCLK_DIV_8: SCLK/8
  *                     PM_BUFFER_SCLK_DIV_16: SCLK/16
  *                     PM_BUFFER_SCLK_DIV_32: SCLK/32
  *                     PM_BUFFER_SCLK_DIV_64: SCLK/64
  *                     PM_BUFFER_SCLK_DIV_128: SCLK/128
  *   ޡ
  */
void
PM_Internal_Test_Buffer_Clock_Scaler_Config (uint32_t SclkScaler)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_BUFFER_SCLK_SCALER(SclkScaler));

    /*---------------- PM_CTL1ĴBUFCLKDIVλ ----------------*/
    PM_CTL1 = SFR_Config (PM_CTL1, ~PM_CTL1_BUFCLKDIV, SclkScaler);
}

/**
  *   BAT_BORѹѡ
  *   Voltage: BAT_BORѹѡȡֵΪ:
  *                  PM_BATTERY_VOLTAGE_1P6V: 1.6V
  *                  PM_BATTERY_VOLTAGE_1P8V: 1.8V
  *                  PM_BATTERY_VOLTAGE_2P06V: 2.06V
  *                  PM_BATTERY_VOLTAGE_2P4V: 2.4V
  *                  PM_BATTERY_VOLTAGE_2P88V: 2.88V
  *                  PM_BATTERY_VOLTAGE_3P6V: 3.6V
  *   ޡ
  */
void
PM_Battery_BOR_Config (uint32_t Voltage)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_BATTERY_VOLTAGE(Voltage));

    /*---------------- PM_CTL1ĴBATBORSELλ ----------------*/
    PM_CTL1 = SFR_Config (PM_CTL1, ~PM_CTL1_BATBORSEL, Voltage);
}

/**
  *   BAT_BORʹܡ
  *   NewState: BAT_BORʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Battery_BOR_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴBATBORENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹBAT_BOR */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_BATBOREN_POS);
    }
    else
    {
        /* ֹBAT_BOR */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_BATBOREN_POS);
    }
}

/**
  *   PVM1ʹܡ
  *   NewState: PVM1ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Peripheral_Voltage_Monitoring_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴVREFENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܲοѹģ */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
    else
    {
        /* ֹοѹģ */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_VREFEN_POS);
    }
}

/**
  *   õѹѡ
  *   Voltage: ѹѡȡֵΪ:
  *                  PM_VOLTAGE_DETECTION_2P1V: 2.1V
  *                  PM_VOLTAGE_DETECTION_2P25V: 2.25V
  *                  PM_VOLTAGE_DETECTION_2P4V: 2.4V
  *                  PM_VOLTAGE_DETECTION_2P55V: 2.55V
  *                  PM_VOLTAGE_DETECTION_2P7V: 2.7V
  *                  PM_VOLTAGE_DETECTION_2P85V: 2.85V
  *                  PM_VOLTAGE_DETECTION_2P95V: 2.95V
  *   ޡ
  */
void
PM_Voltage_Detection_Config (uint32_t Voltage)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_VOLTAGE_DETECTION(Voltage));

    /*---------------- PM_CTL1ĴPVDSλ ----------------*/
    PM_CTL1 = SFR_Config (PM_CTL1, ~PM_CTL1_PVDS, Voltage);
}

/**
  *   LPRѡ
  *   LPRMODE: LPRѡȡֵΪ:
  *                  PM_LPR_MODE_CAPLESS
  *                  PM_LPR_MODE_CAP
  *   ޡ
  */
void
PM_BKP_LPR_Type_Select(uint32_t LPRMODE)
{
	/* У */
	CHECK_RESTRICTION(CHECK_PM_BKPLPR_MODE(LPRMODE));

	PM_CTL1 = SFR_Config(PM_CTL1, ~PM_CTL1_BKPLPRTYPE, LPRMODE << PM_CTL1_BKPLPRTYPE_POS);
}

/**
  *  õ͹ģʽ DEBUG ʹܡ
  *   NewState: ȡֵΪ:
  *                TRUE: ʹܵ͹ģʽ DEBUG 
  *                FALSE: ֹ͹ģʽ DEBUG 
  *   ޡ
  */
void
PM_LP_DEBUG_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL1 = SFR_Config(PM_CTL1, ~PM_CTL1_LPDEBUG, NewState << PM_CTL1_LPDEBUG_POS);
}

/**
  *  LVL_LDO18 ǿʹܡ
  *   NewState: ȡֵΪ:
  *                TRUE: LVL_LDO18 ǿʹ
  *                FALSE: 
  *   ޡ
  */
void
PM_LVL_LDO18_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL1 = SFR_Config(PM_CTL1, ~PM_CTL1_P18LVLSENB, (!NewState) << PM_CTL1_P18LVLSENB_POS);
}

/**
  *  RAM  ECC ʹʹܡ
  *   NewState: ȡֵΪ:
  *                TRUE: ʹ RAM  ECC 
  *                FALSE: ֹ RAM  ECC 
  *   ޡ
  */
void
PM_RAM_ECC_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL1 = SFR_Config(PM_CTL1, ~PM_CTL1_DPRAMECCEN, NewState << PM_CTL1_DPRAMECCEN_POS);
}

/**
  *  ͹ CAN ʱѡ
  *   CLK: ȡֵΪ:
  *                PM_LPCAN_SCK_INTLF: ѡڲƵ INTLF Ϊʱ
  *                PM_LPCAN_SCK_LP4M: ѡ LP4M Ϊʱ
  *   ޡ
  */
void
PM_LPCAN_Work_CLK_Select(uint32_t CLK)
{
	/* У */
	CHECK_RESTRICTION(CHECK_PM_LPCAN_SCK(CLK));

	PM_CTL1 = SFR_Config(PM_CTL1, ~PM_CTL1_LPCANCLKS, CLK << PM_CTL1_LPCANCLKS_POS);
}

/**
  *   õѹ⹦ʹܡ
  *   NewState: ѹ⹦ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Voltage_Detection_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴPVDENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܵѹ⹦ */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_PVDEN_POS);
    }
    else
    {
        /* ֹѹ⹦ */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_PVDEN_POS);
    }
}

/**
  *   ⲿWKPxʹܡ
  *   PinSel: ţȡֵΪ
  *                 PM_PIN_WKP1: WKP1
  *                 PM_PIN_WKP2: WKP2
  *                 PM_PIN_WKP3: WKP3
  *                 PM_PIN_WKP4: WKP4
  *                 PM_PIN_WKP5: WKP5
  *       NewState: ⲿWKPxʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_External_Wakeup_Pin_Enable (uint32_t PinSel, FunctionalState NewState)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_PIN_WKP(PinSel));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL2ĴWKPxENλ ----------------*/
    tmpreg = PM_CTL2_WKP1EN << PinSel;
    if (NewState != FALSE)
    {
        /* ʹWKPxƽ仯 */
        PM_CTL2 |= tmpreg;
    }
    else
    {
        /* ֹWKPxƽ仯 */
        PM_CTL2 &= ~tmpreg;
    }
}

/**
  *   ⲿWKPxѴء
  *   PinSel: ţȡֵΪ
  *                 PM_PIN_WKP1: WKP1
  *                 PM_PIN_WKP2: WKP2
  *                 PM_PIN_WKP3: WKP3
  *                 PM_PIN_WKP4: WKP4
  *                 PM_PIN_WKP5: WKP5
  *       TriggerEdge: ⲿWKPxѴأȡֵΪ:
  *                      PM_TRIGGER_RISE_EDGE: WKPxش
  *                      PM_TRIGGER_FALL_EDGE: WKPx½ش
  *   ޡ
  */
void
PM_External_Wakeup_Edge_Config (uint32_t PinSel, uint32_t TriggerEdge)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_PIN_WKP(PinSel));
    CHECK_RESTRICTION(CHECK_PM_TRIGGER_EDGE(TriggerEdge));

    /*---------------- PM_CTL2ĴWKP5Pλ ----------------*/
    tmpreg = PM_CTL2_WKP1P << PinSel;
    if (TriggerEdge != PM_TRIGGER_RISE_EDGE)
    {
        /* WKPx½ش */
        PM_CTL2 |= tmpreg;
    }
    else
    {
        /* WKPxش */
        PM_CTL2 &= ~tmpreg;
    }
}

/**
  *   òͬ蹤StopģʽڲƵʱʹܡ
  *   Peripheral: ѡȡֵΪ
  *                     PM_PERIPHERAL_CAN6: CAN6
  *                     PM_PERIPHERAL_CCP: CCP
  *                     PM_PERIPHERAL_EWM: EWM
  *                     PM_PERIPHERAL_USART0: USART0
  *       NewState: ڲƵʱʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Stop_Mode_Peripheral_INLF_Enable (uint32_t Peripheral,
                    FunctionalState NewState)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_PERIPHERAL(Peripheral));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL2ĴxxxxCLKLPENλ ----------------*/
    tmpreg = PM_CTL2_CAN6CLKLPEN << Peripheral;
    if (NewState != FALSE)
    {
        /* ڲƵʱΪ蹤ʱ */
        PM_CTL2 |= tmpreg;
    }
    else
    {
        /* ֹʱ */
        PM_CTL2 &= ~tmpreg;
    }
}

/**
  *   òͬ踴λ
  *   Peripheral: ѡȡֵΪ
  *                     PM_PERIPHERAL_CAN6: CAN6
  *                     PM_PERIPHERAL_CCP: CCP
  *                     PM_PERIPHERAL_EWM: EWM
  *                     PM_PERIPHERAL_USART0: USART0
  *       ResetStatus: 踴λ״̬ȡֵΪ:
  *                      PERIPHERAL_RST_STATUS: 账ڸλ״̬
  *                      PERIPHERAL_OUTRST_STATUS: ˳λ
  *   ޡ
  */
void
PM_Peripheral_Reset_Config (uint32_t Peripheral, uint32_t ResetStatus)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_PERIPHERAL(Peripheral));
    CHECK_RESTRICTION(CHECK_PERIPHERAL_RST(ResetStatus));

    /*---------------- PM_CTL2ĴxxxxLPENλ ----------------*/
    tmpreg = PM_CTL2_CAN6LPEN << Peripheral;
    if (ResetStatus != PERIPHERAL_RST_STATUS)
    {
        /* ģ˳λ */
        PM_CTL2 |= tmpreg;
    }
    else
    {
        /* ģ鴦ڸλ״̬ */
        PM_CTL2 &= ~tmpreg;
    }
}

/**
  *   PMC ״̬жʹܡ
  *   NewState: ȡֵΪ
  *                TRUE  ʹ PMC ״̬ж
  *                FALSE ֹ PMC ״̬ж
  *   ޡ
  */
void
PM_PMC_Error_Status_INT_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_PMCIE, NewState << PM_CTL2_PMCIE_POS);
}

/**
  *   QEI0 λλ
  *   NewState: ȡֵΪ
  *                TRUE   QEI0 ģ˳λ
  *                FALSE QEI0 ģ鴦ڸλ״̬
  *   ޡ
  */
void
PM_QEI0_Reset_Control(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_QEI0LPEN, NewState << PM_CTL2_QEI0LPEN_POS);
}

/**
  *   GPIOA λλ
  *   NewState: ȡֵΪ
  *                TRUE  GPIOA ģ˳λ
  *                FALSE GPIOA ģ鴦ڸλ״̬
  *   ޡ
  */
void
PM_GPIOA_Reset_Control(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_GPIOALPEN, NewState << PM_CTL2_GPIOALPEN_POS);
}

/**
  *   RTC λλ
  *   NewState: ȡֵΪ
  *                TRUE  RTC ģ˳λ
  *                FALSE RTC ģ鴦ڸλ״̬
  *   ޡ
  */
void
PM_RTC_Reset_Control(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_RTCLPEN, NewState << PM_CTL2_RTCLPEN_POS);
}

/**
  *   QEI0  Stop ģʽڲƵʱʹܡ
  *   NewState: ȡֵΪ
  *                TRUE  ڲƵʱΪ QEI0 ʱ
  *                FALSE ֹʱ
  *   ޡ
  */
void
PM_QEI0_Work_Stop_INTLF_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_QEI0CLKLPEN, NewState << PM_CTL2_QEI0CLKLPEN_POS);
}

/**
  *   RTC  Stop ģʽڲƵʱʹܡ
  *   NewState: ȡֵΪ
  *                TRUE  ڲƵ/ⲿƵʱΪ RTC ʱ
  *                FALSE ֹʱ
  *   ޡ
  */
void
PM_RTC_Work_Stop_INTLF_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CTL2 = SFR_Config(PM_CTL2, ~PM_CTL2_RTCCLKLPEN, NewState << PM_CTL2_RTCCLKLPEN_POS);
}


#if _CLOSE_SERVICE_
/**
  *   ԴPORǿʹܡ
  *   NewState: ԴPORǿʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void PM_Vdd_Por_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴVDDPORONλ ----------------*/
    if (NewState != FALSE)
    {
        /*  shutdownģʽǿʹVDD_POR */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_VDDPORON_POS);
    }
    else
    {
        /* shutdownģʽӲԶرVDD_POR */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_VDDPORON_POS);
    }
}

/**
  *   LP_BGʹܡ
  *   NewState: LP_BGʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Low_Power_Bandgap_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL0ĴLPBGONλ ----------------*/
    if (NewState != FALSE)
    {
        /*  ʹLP_BG */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_LPBGON_POS);
    }
    else
    {
        /* ʹLP_BGBORLPRPVDLCDpumpʹʱLP_BGԶʹ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_LPBGON_POS);
    }
}

/**
  *   ùģʽʱѡ
  *   Mode: ģʽʱѡȡֵΪ:
  *               PM_POWER_DISSIPATION_SCHEME1: Scheme1ʱ
  *               PM_POWER_DISSIPATION_SCHEME2: Scheme2ʱο4003ϸ
  *   ޡ
  */
void
PM_Power_Dissipation_Mode_Config (uint32_t Mode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_POWER_DISSIPATION(Mode));

    /*---------------- PM_CTL0ĴPDMSELλ ----------------*/
    if (Mode != PM_POWER_DISSIPATION_SCHEME1)
    {
        /* Scheme2ʱο4003ϸ */
        SFR_SET_BIT_ASM(PM_CTL0, PM_CTL0_PDMSEL_POS);
    }
    else
    {
        /* Scheme1ʱ */
        SFR_CLR_BIT_ASM(PM_CTL0, PM_CTL0_PDMSEL_POS);
    }
}

/**
  *   POR12رյLDOرյʱʱ䡣
  *   DelayTime: POR12رյLDOرյʱʱãȡֵΪ:
  *                    PM_POWER_DISSIPATION_500ns: 500ns
  *                    PM_POWER_DISSIPATION_500ns_1T: 500ns+1* T(INTLFOSC)
  *                    PM_POWER_DISSIPATION_500ns_2P5T: 500ns+2.5* T(INTLFOSC)
  *   ޡ
  */
void
PM_Power_Dissipation_Mode_Delay_Config (uint32_t DelayTime)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_POWER_DISSIPATION_DELAY(DelayTime));

    /*---------------- PM_CTL0ĴPDMDELAYλ ----------------*/
    PM_CTL0 = SFR_Config (PM_CTL0, ~PM_CTL0_PDMDELAY, DelayTime);
}

/**
  *   ڲBUFFERʹܡ
  *   NewState: ڲBUFFERʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Internal_Test_Buffer_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CTL1ĴINTBUFENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹڲBUFFER */
        SFR_SET_BIT_ASM(PM_CTL1, PM_CTL1_INTBUFEN_POS);
    }
    else
    {
        /* ֹڲBUFFER */
        SFR_CLR_BIT_ASM(PM_CTL1, PM_CTL1_INTBUFEN_POS);
    }
}
#endif /* _CLOSE_SERVICE_ */
/**
  *   ##### Դ(PM)ƹܺ #####
  */


/**
  *   ##### Դ(PM)״̬־ #####
  */
/**
  *   㸴λ/״̬־λ
  *   EventSel: λ/¼ȡֵΪ
  *                   PM_RESET_IWDT: Źλ¼
  *                   PM_WAKEUP_EXTERNAL_PIN: ⲿŻ¼
  *                   PM_RESET_POR: PORλ¼
  *                   PM_RESET_BOR: BORλ¼
  *                   PM_RESET_SOFTWARE: λ¼
  *   ޡ
  */
void
PM_Clear_Reset_And_Wakeup_Flag (uint32_t EventSel)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CLEAR_STA0_RESET_WAKEUP(EventSel));
    tmpmask = (uint32_t)1 << EventSel;
    /*---------------- PM_STA0ĴRSTFCλ ----------------*/
    SFR_SET_BIT_ASM(PM_STA0, PM_STA0_RSTFC_POS);
    while(PM_STA0 & tmpmask);
    SFR_CLR_BIT_ASM(PM_STA0, PM_STA0_RSTFC_POS);
}

/**
  *   ȡӦλ/¼״̬־
  *   EventSel: λ/¼ȡֵΪ
  *                   PM_RESET_IWDT: Źλ¼
  *                   PM_WAKEUP_EXTERNAL_PIN: ⲿŻ¼
  *                   PM_RESET_POR: PORλ¼
  *                   PM_RESET_BOR: BORλ¼
  *                   PM_RESET_SOFTWARE: λ¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP5: WKP5Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP4: WKP4Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP3: WKP3Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP2: WKP2Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP1: WKP1Ż¼
  *   1: ˸λ/¼
  *       0: δλ/¼
  */
FlagStatus
PM_Get_Reset_Flag (uint32_t EventSel)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_RESET_WAKEUP(EventSel));

    if (EventSel < 32)
    {
        tmpreg = PM_STA0;
        tmpmask = (uint32_t)1 << EventSel;
    }
    else
    {
        tmpreg = PM_STA1;
        tmpmask = (uint32_t)1 << (EventSel - 32);
    }

    /*---------------- ȡPM_STAxĴӦλ ----------------*/
    if (tmpreg & tmpmask)
    {
        /* ˸λ/¼ */
        return SET;
    }
    else
    {
        /* δλ/¼ */
        return RESET;
    }
}

/**
  *   ⲿWKPx״̬־
  *   EventSel: λ/¼ȡֵΪ
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP5: WKP5Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP4: WKP4Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP3: WKP3Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP2: WKP2Ż¼
  *                   PM_WAKEUP_EXTERNAL_PIN_WKP1: WKP1Ż¼
  *   ޡ
  */
void
PM_Clear_External_Wakeup_Pin_Flag (uint32_t EventSel)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_CLEAR_STA1_WAKEUP(EventSel));
    tmpmask = (uint32_t)1 << (EventSel - 32);
    /*---------------- PM_STACĴWKPxSCλ ----------------*/
    PM_STAC |= tmpmask;
    while(PM_STA1 & tmpmask);
    PM_STAC &= ~tmpmask;
}

/**
  *   PMCIF ־
  *   
  *   ޡ
  */
void
PM_Clear_PMCIF_Flag(void)
{
	PM_STAC = SFR_Config(PM_STAC, ~PM_STAC_PMCIC, PM_STAC_PMCIC);
}

/**
  *   ȡ͹ģʽ״̬־
  *   ޡ
  *   1: ѹлLPR0: ѹMRʹɡ
  */
FlagStatus
PM_Get_Low_Power_Running_State (void)
{
    /*---------------- ȡPM_STA0ĴLPRUNFλ ----------------*/
    if (PM_STA0 & PM_STA0_LPRUNF)
    {
        /* ѹлLPR */
        return SET;
    }
    else
    {
        /* ѹMRʹ */
        return RESET;
    }
}

/**
  *   ȡ͹ĵѹLPR״̬
  *   ޡ
  *   1: ͹ĵѹLPRʹɣ
  *       0: ͹ĵѹLPRδʹɡ
  */
FlagStatus
PM_Get_LPR_Status (void)
{
    /*---------------- ȡPM_STA0ĴLPRSTAλ ----------------*/
    if (PM_STA0 & PM_STA0_LPRSTA)
    {
        /* ͹ĵѹLPRʹ */
        return SET;
    }
    else
    {
        /* ͹ĵѹLPRδʹ */
        return RESET;
    }
}

/**
  *   ȡѹ״̬
  *   ޡ
  *   1: VDDѹPVDѹ㣻
  *       0: VDDѹPVDѹ㡣
  */
FlagStatus
PM_Get_Peripheral_Voltage_Detection_Status (void)
{
    /*---------------- ȡPM_STA1ĴPVDSTAλ ----------------*/
    if (PM_STA1 & PM_STA1_PVDSTA)
    {
        /* VDDѹPVDѹ */
        return SET;
    }
    else
    {
        /* VDDѹPVDѹ */
        return RESET;
    }
}

/**
  *   ȡPMC жϱ־
  *   ޡ
  *   1:  PMC жϣSTOPERROR  PMCERROR  1
  *       0: á
  */
FlagStatus
PM_Get_PMC_Error_INT_Falg(void)
{
    /*---------------- ȡPM_STA1ĴPMCIFλ ----------------*/
    if (PM_STA1 & PM_STA1_PMCIF)
    {
        /*  PMC ж */
        return SET;
    }
    else
    {
        /*  */
        return RESET;
    }
}

/**
  *   ȡPOR18 ״̬
  *   ޡ
  *   1:  POR18
  *       0: δ POR18
  */
FlagStatus
PM_POR18_Finish_Falg(void)
{
    /*---------------- ȡPM_STA1ĴPOR18FINISHλ ----------------*/
    if (PM_STA1 & PM_STA1_POR18STA)
    {
        /*  POR18 */
        return SET;
    }
    else
    {
        /* δ POR18 */
        return RESET;
    }
}

/**
  *   ȡģʽжϱ־
  *   ޡ
  *   1: ڵ綯ǰ˻
  *       0: ˵ģʽ
  */
FlagStatus
PM_PD_PMC_Error_Falg(void)
{
    /*---------------- ȡPM_STA1ĴPMCERRORλ ----------------*/
    if (PM_STA1 & PM_STA1_PMCERROR)
    {
        /*  POR18 */
        return SET;
    }
    else
    {
        /* δ POR18 */
        return RESET;
    }
}

/**
  *   ȡSTOP ģʽжϱ־
  *   ޡ
  *   1:  STOP ģʽʱڻ
  *       0:  STOP ģʽ
  */
FlagStatus
PM_Stop_Mode_Error_Falg(void)
{
    /*---------------- ȡPM_STA1ĴSTOPERRORλ ----------------*/
    if (PM_STA1 & PM_STA1_STOPERROR)
    {
        /*  STOP ģʽʱڻ */
        return SET;
    }
    else
    {
        /*  STOP ģʽ */
        return RESET;
    }
}


/**
  *   ##### Դ(PM)״̬־ #####
  */


/**
  *   ##### Դ(PM)У׼ƺ #####
  */
/**
  *   ƯУ׼
  *   Calibration: ƯУ׼ȡֵΪ:
  *                      PM_CURRENT_OUTPUT_2uA: 2uA
  *                      PM_CURRENT_REDUCE_10_PERCENT: С10%
  *                      PM_CURRENT_INCREASE_10_PERCENT: 10%
  *                      PM_CURRENT_INCREASE_20_PERCENT: 20%
  *   ޡ
  */
void
PM_Zero_Drift_Current_Config (uint32_t Calibration)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_CURRENT_CONFIG(Calibration));

    /*---------------- PM_CAL0ĴZEROTISTRIMλ ----------------*/
    PM_CAL0 = SFR_Config (PM_CAL0, ~PM_CAL0_ZEROTISTRIM, Calibration);
}

/**
  *   BORѹѡ
  *   Voltage: BORѹѡȡֵΪ:
  *                  PM_BOR_VOLTAGE_2P32V: BORѹΪ2.32V
  *                  PM_BOR_VOLTAGE_2P54V: BORѹΪ2.54V
  *                  PM_BOR_VOLTAGE_2P89V: BORѹΪ2.89V
  *                  PM_BOR_VOLTAGE_3P23V: BORѹΪ3.23V
  *   ޡ
  */
void
PM_BOR_Voltage_Config (uint32_t Voltage)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_BOR_VOLTAGE(Voltage));

    /*---------------- PM_CAL0ĴBORSELλ ----------------*/
    PM_CAL0 = SFR_Config (PM_CAL0, ~PM_CAL0_BORSEL, Voltage);
}

/**
  *   MRMR_HVģУ׼
  *   MRSel: MRMR_HVģѡȡֵΪ
  *                PM_MR_MODULE: MRģ
  *                PM_MR_HV_MODULE: MR_HVģ
  *       Voltage: У׼ѹѡȡֵΪ:
  *                  PM_MR_VOLTAGE_1P2V: BORѹΪ1.2V
  *                  PM_MR_VOLTAGE_0P9V: BORѹΪ0.9V
  *                  PM_MR_VOLTAGE_1V: BORѹΪ1V
  *                  PM_MR_VOLTAGE_1P32V: BORѹΪ1.32V
  *   ޡ
  */
void
PM_Main_Regulator_Voltage_Config (uint32_t MRSel, uint32_t Voltage)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_PM_MR_MODULE(MRSel));
    CHECK_RESTRICTION(CHECK_PM_MR_VOLTAGE(Voltage));

    /*------------- PM_CAL0ĴMRTRIMMRHVTRIMλ -------------*/
    tmpreg = Voltage << MRSel;
    tmpmask = (uint32_t)0xF << MRSel;
    PM_CAL0 = SFR_Config (PM_CAL0, ~tmpmask, tmpreg);
}

/**
  *   MR_HVģʹܡ
  *   NewState: MR_HVģʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
PM_Main_Regulator_HV_Enable (FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- PM_CAL0ĴMRHVENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹMR_HVģ */
        SFR_SET_BIT_ASM(PM_CAL0, PM_CAL0_MRHVEN_POS);
    }
    else
    {
        /* ֹMR_HVģ */
        SFR_CLR_BIT_ASM(PM_CAL0, PM_CAL0_MRHVEN_POS);
    }
}

/**
  *   òͬĻ׼У׼
  *   Reference: ͬĻ׼ѡȡֵΪ
  *                    PM_REFERENCE_LDO12: LDO12׼У׼
  *                    PM_REFERENCE_LDO18: LDO18׼У׼
  *       Calibration: У׼ѹѡ
  *                    PM_REFERENCE_LDO12ȡֵΧΪ0~0x1F,
  *                    PM_REFERENCE_LDO18ȡֵΧΪ0~0x1F
  *   ޡ
  */
void
PM_Reference_Calibration_Config (uint32_t Reference, uint32_t Calibration)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;

    /* У */
    switch (Reference)
    {
        case PM_REFERENCE_LDO12:
            CHECK_RESTRICTION(CHECK_PM_LDO12_CALIBRATION(Calibration));
            tmpreg = Calibration << PM_REFERENCE_LDO12;
            tmpmask = PM_CAL0_BGTRIMMRLDO;
            break;
        case PM_REFERENCE_LDO18:
            CHECK_RESTRICTION(CHECK_PM_LDO18_CALIBRATION(Calibration));
            tmpreg = Calibration << PM_REFERENCE_LDO18;
            tmpmask = PM_CAL0_BGTRIMFLLDO;
            break;
        default:
            CHECK_RESTRICTION(0);
            break;
    }

    /*------------- PM_CAL0ĴӦУ׼λ -------------*/
    PM_CAL0 = SFR_Config (PM_CAL0, ~tmpmask, tmpreg);
}

/**
  *   ڲƵУ׼
  *   Calibration: ƫõѡȡֵΪ
  *                      PM_INTLF_BIAS_CURRENT_10nA: 10nA
  *                      PM_INTLF_BIAS_CURRENT_12P5nA: 12.5nA
  *                      PM_INTLF_BIAS_CURRENT_15nA: 15nA
  *                      PM_INTLF_BIAS_CURRENT_17P5nA: 17.5nA
  *                      PM_INTLF_BIAS_CURRENT_0nA: 0nA
  *                      PM_INTLF_BIAS_CURRENT_2P5nA: 2.5nA
  *                      PM_INTLF_BIAS_CURRENT_5nA: 5nA
  *                      PM_INTLF_BIAS_CURRENT_7P5nA: 7.5nA
  *   ޡ
  */
void
PM_INTLF_Bias_Current_Config (uint32_t Calibration)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_INTLF_BIAS_CURRENT(Calibration));

    /*------------- PM_CAL1ĴLSITRIMBIAS -------------*/
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LSI_TRIM_BIAS, Calibration);
}

/**
  *   ⲿƵУ׼
  *   Calibration: ƫõѡȡֵΪ
  *                      PM_EXTLF_BIAS_CURRENT_20nA: 20nA
  *                      PM_EXTLF_BIAS_CURRENT_25nA: 25nA
  *                      PM_EXTLF_BIAS_CURRENT_30nA: 30nA
  *                      PM_EXTLF_BIAS_CURRENT_35nA: 35nA
  *                      PM_EXTLF_BIAS_CURRENT_0nA: 0nA
  *                      PM_EXTLF_BIAS_CURRENT_5nA: 5nA
  *                      PM_EXTLF_BIAS_CURRENT_10nA: 10nA
  *                      PM_EXTLF_BIAS_CURRENT_15nA: 15nA
  *   ޡ
  */
void
PM_EXTLF_Bias_Current_Config (uint32_t Calibration)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_EXTLF_BIAS_CURRENT(Calibration));

    /*------------- PM_CAL1ĴLSETRIMBIAS -------------*/
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LSE_TRIMBIAS, Calibration);
}

/**
  *   ڲƵУ׼
  *   Calibration: У׼ֵȡֵΧΪ0~0x1F
  *   ޡ
  */
void
PM_INTLF_Capacitance_Calibration_Config (uint32_t Calibration)
{
	uint32_t tmpreg=0;
    /* У */
    CHECK_RESTRICTION(CHECK_PM_CAPACITANCE_CALIBRATION(Calibration));

    /*------------- PM_CAL1ĴLSITRIMCAP -------------*/
    tmpreg = Calibration << PM_CAL1_LSI_TRIM_CAP0_POS;
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LSI_TRIM_CAP, tmpreg);
}

/**
  *   õ͹ƫõУ׼
  *   Calibration: У׼ֵȡֵΧΪ0~0x7
  *   ޡ
  */
void
PM_LP_Bias_Calibration_Config (uint32_t Calibration)
{
	uint32_t tmpreg=0;
    /* У */
    CHECK_RESTRICTION(CHECK_PM_LP_BIAS_CALIBRATION(Calibration));

    /*------------- PM_CAL1ĴLPTRIMBIAS -------------*/
    tmpreg = Calibration << PM_CAL1_LP_TRIM_BIAS0_POS;
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LP_TRIM_BIAS, tmpreg);
}

/**
  *   ڲƵУ׼
  *   Calibration: У׼ֵȡֵΧΪ0~0x7
  *   ޡ
  */
void
PM_LPBG_Pump_Calibration_Config (uint32_t Calibration)
{
	uint32_t tmpreg=0;
    /* У */
    CHECK_RESTRICTION(CHECK_PM_PUMP_CALIBRATION(Calibration));

    /*------------- PM_CAL1ĴLPBGTRIMPUMP -------------*/
    tmpreg = Calibration << PM_CAL1_LPBG_TRIM_PUMP0_POS;
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LPBG_TRIM_PUMP, tmpreg);

}

/**
  *   ⲿƵNƫУ׼
  *   Calibration: ѡȡֵΪ
  *                      PM_BRANCH_CURRENT_NONE: û֧·
  *                      PM_BRANCH_CURRENT_50_PERCENT: 50%֧·
  *                      PM_BRANCH_CURRENT_150_PERCENT: 150%֧·
  *                      PM_BRANCH_CURRENT_200_PERCENT: 200%֧·
  *   ޡ
  */
void
PM_EXTLF_N_Bias_Current_Config (uint32_t Calibration)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_BRANCH_CURRENT(Calibration));

    /*------------- PM_CAL1ĴLSETRIMNBIAS -------------*/
    PM_CAL1 = SFR_Config (PM_CAL1, ~PM_CAL1_LSE_TRIM_NBIAS, Calibration);
}

/**
  *   ⲿƵѡλ
  *   PeripheralPort: ⲿƵIOڣȡֵΪ:
  *                         PM_EXTLF_PIN1_IO_PORT: ѡⲿƵ1
  *                         PM_EXTLF_PIN2_IO_PORT: ѡⲿƵ2
  *   ޡ
  */
void
PM_EXTLF_PIN_Selection_Config (uint32_t PeripheralPort)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_EXTLF_PIN_IO(PeripheralPort));

    /*---------------- PM_CTL1ĴLSESELλ ----------------*/
    if (PeripheralPort != PM_EXTLF_PIN1_IO_PORT)
    {
        /* ѡⲿƵ2*/
        SFR_SET_BIT_ASM(PM_CAL1, PM_CAL1_LSE_SEL_POS);
    }
    else
    {
        /* ѡⲿƵ1 */
        SFR_CLR_BIT_ASM(PM_CAL1, PM_CAL1_LSE_SEL_POS);
    }
}

/**
  *   ⲿƵѡλ
  *   PeripheralPort: ⲿƵIOڣȡֵΪ:
  *                         PM_EXTHF_PIN1_IO_PORT: ѡⲿƵ1
  *                         PM_EXTHF_PIN2_IO_PORT: ѡⲿƵ2
  *   ޡ
  */
void
PM_EXTHF_PIN_Selection_Config (uint32_t PeripheralPort)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_EXTHF_PIN_IO(PeripheralPort));

    /*---------------- PM_CTL1ĴHSESELλ ----------------*/
    if (PeripheralPort != PM_EXTHF_PIN1_IO_PORT)
    {
        /* ѡⲿƵ2*/
        SFR_SET_BIT_ASM(PM_CAL1, PM_CAL1_HSE_SEL_POS);
    }
    else
    {
        /* ѡⲿƵ1 */
        SFR_CLR_BIT_ASM(PM_CAL1, PM_CAL1_HSE_SEL_POS);
    }
}

/**
  *   LPRģѡ
  *   ModeSel: LPRģѡȡֵΪ:
  *                  PM_LPR_DEFAULT: ͹ģʽ
  *                  PM_LPR_BACKUP: ߹ģʽ
  *   ޡ
  */
void
PM_LPR_Module_Config (uint32_t ModeSel)
{
    /* У */
    CHECK_RESTRICTION(CHECK_PM_LPR_MODE(ModeSel));

    /*---------------- PM_CAL0ĴMRBGSELλ ----------------*/
    if (ModeSel != PM_LPR_DEFAULT)
    {
        /* ߹ģʽ */
        SFR_SET_BIT_ASM(PM_CAL0, PM_CAL0_MRBGSEL_POS);
    }
    else
    {
        /* ͹ģʽ */
        SFR_CLR_BIT_ASM(PM_CAL0, PM_CAL0_MRBGSEL_POS);
    }
}

/**
  *   ⲿλ NRST1PH7NRST0PD0 ʹܡ
  *   NRSTPIN: λѡȡֵΪ:PM_NRST0PM_NRST1
  * 	 NewState: TURE ʹ\FALSE ֹ
  *   ޡ
  */
void
PM_Reset_Pin_Config(uint32_t NRSTPIN, FunctionalState NewState)
{
	uint32_t tmpreg = 0;
	uint32_t tmask = 0;

	/* У */
	CHECK_RESTRICTION(CHECK_PM_NRST_PIN(NRSTPIN));
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	tmpreg = NewState << (PM_CAL0_NRST0EN_POS + NRSTPIN);
	tmask = 1 << (PM_CAL0_NRST0EN_POS + NRSTPIN);

	PM_CAL0 = SFR_Config(PM_CAL0, ~tmask, tmpreg);
}

/**
  *   IWDT ʹܡ
  *   NewState:
  * 			TURE ʹ IWDT
  *				FALSE 
  *   
  * עʹܺҲԶʹڲƵ INTLF Ҳ IWDT λʱλ IWDT 
  */
void
PM_IWDT_Enable(FunctionalState NewState)
{
	/* У */
	CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

	PM_CAL0 = SFR_Config(PM_CAL0, ~PM_CAL0_PM_IWDTEN, NewState << PM_CAL0_PM_IWDTEN_POS);
}

/**
  *   LPR12 capless λ
  *   Calibration: Calibration <= 0x1F
  *   
  */
void
PM_LPR12_CAPLESS_I_Config(uint32_t Calibration)
{
	/* У */
	CHECK_RESTRICTION(CHECK_PM_LPR12_CAPLESS_I(Calibration));

	PM_CAL2 = SFR_Config(PM_CAL2, ~PM_CAL2_LPR12_CAPLESS_I, Calibration << PM_CAL2_LPR12_CAPLESS_I0_POS);
}

/**
  *   LPR12 capless Ŀλ
  *   Calibration:
  * 			PM_LPR12_PC_MAX_90uA: MR12capless ʹ󹦺 90uA
  *				PM_LPR12_PC_MIN_20uA: MR12capless ʹС 20uA
  *				PM_LPR12_PC_MEDIUM_40uA: MR12capless ʹеȹ 40uA
  *   
  */
void
PM_MR12_Capless_PC_Config(uint32_t Calibration)
{
	/* У */
	CHECK_RESTRICTION(CHECK_PM_LPR12_PC(Calibration));

	PM_CAL2 = SFR_Config(PM_CAL2, ~PM_CAL2_CAP_FREE_LDO_IS, Calibration << PM_CAL2_CAP_FREE_LDO_IS0_POS);
}

/**
  *   òο BUFFER ׼У׼
  *   Calibration: Calibration <= 0x3F
  *   
  */
void
PM_BGTRIMREBUF_Config(uint32_t Calibration)
{
	/* У */
	CHECK_RESTRICTION(CHECK_PM_BGTRIMREBUF_VALUE(Calibration));

	PM_CAL2 = SFR_Config(PM_CAL2, ~PM_CAL2_BGTRIMREFBUF, Calibration << PM_CAL2_BGTRIMREFBUF0_POS);
}


/**
  *   ##### Դ(PM)У׼ƺ #####
  */

