/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : KF32A156_i2c.c                                 	      *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This file provides functions for internal IC interface*
 *  				    (I2C) peripherals, including:						  *
 *          			+ Internal integrated circuit interface(I2C)		  *
 *          			  initialization function							  *
 *          			+ Internal integrated circuit interface(I2C)		  *
 *          			  function configuration function					  *
 *          			+ Internal integrated circuit interface(I2C)		  *
 *          			  interrupt management function 					  *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/

/******************************************************************************
**                      		Include Files                                **
******************************************************************************/


#include "kf32a156_i2c.h"
#include "kf32a156_rst.h"
#include "kf32a156_pclk.h"


/*******************************************************************************
**                        (I2C) Module private definition                     **
*******************************************************************************/

/* I2C_CTLRĴ */
#define I2C_CTLR_INIT_MASK              (I2C_CTLR_SMBUS \
                                       | I2C_CTLR_I2CCKS \
                                       | I2C_CTLR_BADR10 \
                                       | I2C_CTLR_SMBT \
                                       | I2C_CTLR_ACKEN \
                                       | I2C_CTLR_ACKDT)


/**
  *   ##### ڲɵ·ӿ(I2C)ʼ #####
  */
/**
  *   I2C踴λ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Reset (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    if (I2Cx == I2C0_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C0RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C0RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_I2C0CLKEN, TRUE);
    }
    else if (I2Cx == I2C1_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C1RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C1RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_I2C1CLKEN, TRUE);
    }
    else if (I2Cx == I2C2_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C2RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_I2C2RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_I2C2CLKEN, TRUE);
    }
    else if (I2Cx == I2C3_SFR)
    {
        RST_CTL3_Peripheral_Reset_Enable(RST_CTL3_I2C3RST, TRUE);
        RST_CTL3_Peripheral_Reset_Enable(RST_CTL3_I2C3RST, FALSE);
        PCLK_CTL3_Peripheral_Clock_Enable(PCLK_CTL3_I2C3CLKEN, TRUE);
    }
    else
    {
        ;
    }
}

/**
  *   I2Cá
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       i2cInitStruct: I2CģϢṹָ롣
  *   ޡ
  */
void
I2C_Configuration (I2C_SFRmap* I2Cx, I2C_InitTypeDef* i2cInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_MODE(i2cInitStruct->m_Mode));
    CHECK_RESTRICTION(CHECK_I2C_CLK(i2cInitStruct->m_ClockSource));
    CHECK_RESTRICTION(CHECK_I2C_BUFR_ADDRESS(i2cInitStruct->m_BADR10));
    CHECK_RESTRICTION(CHECK_SMBUS_MODE(i2cInitStruct->m_MasterSlave));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(i2cInitStruct->m_AckEn));
    CHECK_RESTRICTION(CHECK_I2C_ACKDATA(i2cInitStruct->m_AckData));

    /*---------------------------- I2Cx_CTLRĴ  -----------------*/
    /* ݽṹԱm_ModeSMBUSλ */
    /* ݽṹԱm_ClockSourceI2CCKSλ */
    /* ݽṹԱm_BADR10BADR10λ */
    /* ݽṹԱm_MasterSlaveSMBTλ */
    /* ݽṹԱm_AckEnACKENλ */
    /* ݽṹԱm_AckDataACKDTλ */
	tmpreg = ((i2cInitStruct->m_Mode)    \
			| (i2cInitStruct->m_ClockSource)   \
			| (i2cInitStruct->m_BADR10)  \
			| (i2cInitStruct->m_MasterSlave) \
			| (i2cInitStruct->m_AckEn << I2C_CTLR_ACKEN_POS) \
			| (i2cInitStruct->m_AckData));
    I2Cx->CTLR = SFR_Config (I2Cx->CTLR, ~I2C_CTLR_INIT_MASK, tmpreg);

    /*---------------------------- I2Cx_BRGRĴ  -----------------*/
    /* ݽṹԱm_BaudRateLI2CBRGLλ */
    /* ݽṹԱm_BaudRateHI2CBRGHλ */
    tmpreg = ((i2cInitStruct->m_BaudRateL << I2C_BRGR_I2CBRGL0_POS) \
            | (i2cInitStruct->m_BaudRateH << I2C_BRGR_I2CBRGH0_POS));
    I2Cx->BRGR = SFR_Config (I2Cx->BRGR,
                        ~(I2C_BRGR_I2CBRGL | I2C_BRGR_I2CBRGH),
                        tmpreg);
}

/**
  *   ʼI2CϢṹ塣
  *   dacInitStruct: ָʼĽṹָ롣
  *   ޡ
  */
void I2C_Struct_Init (I2C_InitTypeDef* I2C_InitStruct)
{
    /* ʼ I2Cģʽѡ*/
    I2C_InitStruct->m_Mode = I2C_MODE_I2C;

    /* ʼ I2Cʱѡ */
    I2C_InitStruct->m_ClockSource = I2C_CLK_SCLK;

    /* ʼ I2Cַѡ */
    I2C_InitStruct->m_BADR10 = I2C_BUFRADDRESS_7BIT;

    /* ʼ SMBusѡ */
    I2C_InitStruct->m_MasterSlave = I2C_MODE_SMBUSDEVICE;

    /* ʼ SCL͵ƽռõʱ */
    I2C_InitStruct->m_BaudRateL = 4;

    /* ʼ SCLߵƽռõʱ */
    I2C_InitStruct->m_BaudRateH = 4;

    /* ʼ I2CӦʹ */
    I2C_InitStruct->m_AckEn = FALSE;

    /* ʼ I2CӦλ */
    I2C_InitStruct->m_AckData = I2C_ACKDATA_ACK;
}
/**
  *   ##### ڲɵ·ӿ(I2C)ʼ #####
  */


/**
  *   ##### ڲɵ·ӿ(I2C)ú #####
  */
/**
  *   I2Cʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_Cmd(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴI2CENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2C */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_I2CEN_POS);
    }
    else
    {
        /* ֹI2C */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_I2CEN_POS);
    }
}

/**
  *   I2C10λַʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CʹλϢ
  *                 ȡֵΪ I2C_BUFRADDRESS_10BIT  I2C_BUFRADDRESS_7BIT
  *   ޡ
  */
void
I2C_Bufr_Address_Config(I2C_SFRmap* I2Cx, uint32_t NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_BUFR_ADDRESS(NewState));

    /*---------------- I2C_CTLRĴBADR10λ ----------------*/
    if (NewState != I2C_BUFRADDRESS_7BIT)
    {
        /* I2Cģʹ10λַ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_BADR10_POS);
    }
    else
    {
        /* I2Cģʹ7λַ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_BADR10_POS);
    }
}

/**
  *   I2C_ʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_Generate_START(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴSENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2C */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SEN_POS);
    }
    else
    {
        /* ֹI2C */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SEN_POS);
    }
}

/**
  *   I2Cֹͣʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CֹͣʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_Generate_STOP(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴPENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2Cֹͣ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_PEN_POS);
    }
    else
    {
        /* ֹI2Cֹͣ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_PEN_POS);
    }
}

/**
  *   I2C_AckӦʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CӦʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_Ack_Config (I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴACKENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2C_Ack */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ACKEN_POS);
    }
    else
    {
        /* ֹI2C_Ack */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ACKEN_POS);
    }
}

/**
  *   I2C_AckӦλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CӦλϢȡֵΪ
  *                   I2C_ACKDATA_ACK:  Ӧ
  *                   I2C_ACKDATA_NO_ACK: Ӧ
  *   ޡ
  */
void
I2C_Ack_DATA_Config (I2C_SFRmap* I2Cx, uint32_t NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_ACKDATA(NewState));

    /*---------------- I2C_CTLRĴACKDTλ ----------------*/
    if (NewState != I2C_ACKDATA_ACK)
    {
        /* Ӧ*/
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ACKDT_POS);
    }
    else
    {
        /* Ӧ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ACKDT_POS);
    }
}

/**
  *   I2C_Callʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2C_CallʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_Call_Cmd (I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴGCENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2C_Call */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_GCEM_POS);
    }
    else
    {
        /* ֹI2C_Call */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_GCEM_POS);
    }
}

/**
  *   I2CʱѡĴ
  *   ClkSource: ʱѡȡֵΧΪ
  *                    I2C_CLK_SCLK: ѡSCLKΪI2Cʱ
  *                    I2C_CLK_HFCLK: ѡHFCLKΪI2Cʱ
  *                    I2C_CLK_LFCLK: ѡLFCLKΪI2Cʱ
  *   ޡ
  */
void
I2C_Clock_Config (I2C_SFRmap* I2Cx,uint32_t ClkSource)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_CLK(ClkSource));

    /*---------------- USART_CTLRĴI2CCKSλ ----------------*/
    I2Cx->CTLR = SFR_Config (I2Cx->CTLR, ~I2C_CTLR_I2CCKS, ClkSource);
}

/**
  *   I2CַĴƥλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2C_ַĴƥλȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_MATCH_ADDRESS_Config (I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- USART_CTLRĴMTHALLλ ----------------*/
    if (NewState != FALSE)
    {
        /* ܵĵַϲжϣʹϵͨ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_MTHALL_POS);
    }
    else
    {
        /* ֻI2CADDRַƥʱŻ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_MTHALL_POS);
    }
}

/**
  *   I2C SCLʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2C_SCLʹܣȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_SCL_Enable (I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴENASCLλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹSCL */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ENASCL_POS);
    }
    else
    {
        /* SCLǿΪ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ENASCL_POS);
    }
}

/**
  *   I2C ģʽʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2C_ģʽʹλȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_NMENA_Enable(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴNMENAλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܼģʽ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_NMENA_POS);
    }
    else
    {
        /* ʹܼģʽ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_NMENA_POS);
    }
}

/**
  *   SMBusģʽʹźš
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBusģʽʹźţȡֵΪ:
  *                   I2C_MODE_SMBUS: SMBusģʽ
  *                   I2C_MODE_I2C: I2Cģʽ
  *   ޡ
  */
void
I2C_SMBUS_Enable(I2C_SFRmap* I2Cx, uint32_t NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_MODE(NewState));

    /*---------------- I2C_CTLRĴSMBUSλ ----------------*/
    if (NewState != I2C_MODE_I2C)
    {
        /* 1 = SMBusģʽ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SMBUS_POS);
    }
    else
    {
        /* 0 = I2Cģʽ */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SMBUS_POS);
    }
}

/**
  *   SMBUS͡
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBUSͣȡֵΪ:
  *                   I2C_MODE_SMBUSHOST: SMBus
  *                   I2C_MODE_SMBUSDEVICE: SMBus豸
  *   ޡ
  */
void
I2C_SMBT_Config(I2C_SFRmap* I2Cx, uint32_t NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_SMBUS_MODE(NewState));

    /*---------------- I2C_CTLRĴSMBTλ ----------------*/
    if (NewState != I2C_MODE_SMBUSDEVICE)
    {
        /* SMBus */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SMBT_POS);
    }
    else
    {
        /* SMBus豸*/
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_SMBT_POS);
    }
}

/**
  *   SMBusѡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBusѣȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_SMBus_ALERT_Config(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴALERTλ ----------------*/
    if (NewState != FALSE)
    {
        /* SMBALTʹ */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ALERT_POS);
    }
    else
    {
        /* ͷSMBALTʹ*/
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ALERT_POS);
    }
}

/**
  *   I2Cݡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       Data: дݼĴֵȡֵΪ10λݡ
  *   ޡ
  */
void
I2C_SendData (I2C_SFRmap* I2Cx, uint32_t Data)
{
     /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_BUFR(Data));

    /*---------------- I2C_BUFRĴ ----------------*/
    I2Cx->BUFR = Data;
}

/**
  *   I2C8λݡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       Data: дݼĴֵȡֵΪ8λݡ
  *   ޡ
  */
void
I2C_SendData8 (I2C_SFRmap* I2Cx, uint8_t Data)
{
     /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_BUFR(Data));

    /*---------------- I2C_BUFRĴ ----------------*/
    I2Cx->BUFR = Data;
}

/**
  *   I2Cݡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   32λݵ10λݡ
  */
uint32_t I2C_ReceiveData(I2C_SFRmap* I2Cx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_BUFRĴ ----------------*/
    tmpreg = I2Cx->BUFR;
    tmpreg &= I2C_BUFR_I2CBUF;
    tmpreg >>= I2C_BUFR_I2CBUF0_POS;

    return tmpreg;
}

/**
  *   I2C_ARPʹλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2C_ARPʹλϢȡֵΪ TRUE  FALSE
  *   ޡ
  */
void
I2C_ARP_Enable(I2C_SFRmap* I2Cx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_CTLRĴARPENλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2C_ARP */
        SFR_SET_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ARPEN_POS);
    }
    else
    {
        /* ֹI2C_ARP */
        SFR_CLR_BIT_ASM(I2Cx->CTLR, I2C_CTLR_ARPEN_POS);
    }
}

/**
  *   I2Cַλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       AddrSelect: ַĴѡȡֵΪ0x0~0x3
  *       Data: ַλѡȡֵΪ0x0~0x3FF
  *   ޡ
  */
void
I2C_ADDR_Config(I2C_SFRmap* I2Cx, uint32_t AddrSelect, uint32_t Data)
{
    uint32_t tmpreg = 0;
    uint32_t tmpaddr = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_ADDR_REGISTER(AddrSelect));
    CHECK_RESTRICTION(CHECK_I2C_ADDR(Data));

    /*---------------- I2C_ADDRĴI2CADDλ ----------------*/
    tmpreg = Data << I2C_ADDR_I2CADD0_POS;
    if (0 == AddrSelect)
    {
        tmpaddr = (uint32_t)&(I2Cx->ADDR0);
    }
    else
    {
        tmpaddr = (uint32_t)&(I2Cx->ADDR0) + 4 + (4 * AddrSelect);
    }
    *(volatile uint32_t *)tmpaddr =
                SFR_Config (*(volatile uint32_t *)tmpaddr,
                            ~I2C_ADDR_I2CADD,
                            tmpreg);
//    I2Cx->ADDR0 = SFR_Config (I2Cx->ADDR0, ~I2C_ADDR_I2CADD, tmpreg);
}

/**
  *   I2Cַλ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       AddrSelect: ַĴѡȡֵΪ0x0~0x3
  *       DataMask: ַλѡȡֵΪ0x0~0x3FF
  *   ޡ
  */
void
I2C_MSK_Config(I2C_SFRmap* I2Cx, uint32_t AddrSelect, uint32_t DataMask)
{
    uint32_t tmpreg = 0;
    uint32_t tmpaddr = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_ADDR_REGISTER(AddrSelect));
    CHECK_RESTRICTION(CHECK_I2C_MSK(DataMask));

    /*---------------- I2C_ADDRĴλI2CMSK ----------------*/
    tmpreg = DataMask << I2C_ADDR_I2CMSK0_POS;
    if (0 == AddrSelect)
    {
        tmpaddr = (uint32_t)&(I2Cx->ADDR0);
    }
    else
    {
        tmpaddr = (uint32_t)&(I2Cx->ADDR0) + 4 + (4 * AddrSelect);
    }
    *(volatile uint32_t *)tmpaddr =
                SFR_Config (*(volatile uint32_t *)tmpaddr,
                            ~I2C_ADDR_I2CMSK,
                            tmpreg);
}

/**
  *   I2C SCLߵƽռõʱ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       Period: SCLߵƽռõʱȡֵΪ0x0~0xFFFF
  *   ޡ
  */
void
I2C_BRGH_Config (I2C_SFRmap* I2Cx,uint16_t Period)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_ADDRĴλI2CMSK ----------------*/
    tmpreg = (uint32_t)Period << I2C_BRGR_I2CBRGH0_POS;
    I2Cx->BRGR = SFR_Config (I2Cx->BRGR, ~I2C_BRGR_I2CBRGH, tmpreg);
}

/**
  *   I2C SCL͵ƽռõʱ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       Period: SCL͵ƽռõʱȡֵΪ0x0~0xFFFF
  *   ޡ
  */
void
I2C_BRGL_Config (I2C_SFRmap* I2Cx,uint16_t Period)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_ADDRĴλI2CMSK ----------------*/
    tmpreg = (uint32_t)Period << I2C_BRGR_I2CBRGL0_POS;
    I2Cx->BRGR = SFR_Config (I2Cx->BRGR, ~I2C_BRGR_I2CBRGL, tmpreg);
}

/**
  *   I2CʱӷƵѡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       DIV: I2C_SCK_DIV1 : I2C ʱӷƵΪ 1 1
  *       	   I2C_SCK_DIV2 : I2C ʱӷƵΪ 1 2
  *       	   I2C_SCK_DIV4 : I2C ʱӷƵΪ 1 4
  *   ޡ
  */
void
I2C_CLOCK_Div_Config(I2C_SFRmap* I2Cx, uint32_t DIV)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_I2C_SCK_DIV(DIV));

    tmpreg = DIV << I2C_CTLR_I2CCKDIV0_POS;
    I2Cx->CTLR = SFR_Config(I2Cx->CTLR, ~I2C_CTLR_I2CCKDIV, tmpreg);
}

/**
  *   I2Cݱʱ䡣
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       Time: I2C_Keep_Data_3CLK : 3I2Cʱ
  *       	   	I2C_Keep_Data_4CLK : 4I2Cʱ
  *       	   	I2C_Keep_Data_5CLK : 5I2Cʱ
  *       	   	... ...
  *       	   	I2C_Keep_Data_18CLK : 18I2Cʱ
  *   ޡ
  */
void
I2C_Keep_Data_Time_Config(I2C_SFRmap* I2Cx, uint32_t Time)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
	CHECK_RESTRICTION(CHECK_I2C_KEEP_DATA_CLK(Time));

    tmpreg = Time << I2C_CTLR_DHCNT0_POS;
    I2Cx->CTLR = SFR_Config(I2Cx->CTLR, ~I2C_CTLR_DHCNT, tmpreg);
}

/**
  *   ##### ڲɵ·ӿ(I2C)ú #####
  */


/**
  *   ##### ڲɵ·ӿ(I2C)жϹ #####
  */
/**
  *   I2Cʼźжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2Cʼźжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Start_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴSIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2Cʼźж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_SIE_POS);
    }
    else
    {
        /* ֹI2Cʼźж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_SIE_POS);
    }
}

/**
  *   I2Cֹͣźжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2Cֹͣźжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Stop_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴPIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2Cֹͣźж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_PIE_POS);
    }
    else
    {
        /* ֹI2Cֹͣźж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_PIE_POS);
    }
}

/**
  *   I2CӦжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CӦжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Ack_Fail_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴAFIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2CӦж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_AFIE_POS);
    }
    else
    {
        /* ֹI2CӦж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_AFIE_POS);
    }
}

/**
  *   I2Cʧȥٲжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2Cʧȥٲжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Arbitration_Lost_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴARBLIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2Cʧȥٲж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_ARBLIE_POS);
    }
    else
    {
        /* ֹI2Cʧȥٲж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_ARBLIE_POS);
    }
}

/**
  *   SMBusжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBusжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_SMBus_Alert_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴSMBAIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹSMBusж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_SMBAIE_POS);
    }
    else
    {
        /* ֹSMBusж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_SMBAIE_POS);
    }
}

/**
  *   SMBusͷϵжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBusͷϵжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_SMBus_HostHead_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴSMBHIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹSMBusͷϵж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_SMBHIE_POS);
    }
    else
    {
        /* ֹSMBusͷϵж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_SMBHIE_POS);
    }
}

/**
  *   SMBus豸Ĭϵַжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: SMBus豸Ĭϵַжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_SMBus_Device_Defaultaddress_INT_Enable (I2C_SFRmap* I2Cx,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴSMBDIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹSMBus豸Ĭϵַж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_SMBDIE_POS);
    }
    else
    {
        /* ֹSMBus豸Ĭϵַж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_SMBDIE_POS);
    }
}

/**
  *   I2Cжźʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2Cжź״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_ISIE_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴISIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2Cжź */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_ISIE_POS);
    }
    else
    {
        /* ֹI2Cжź */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_ISIE_POS);
    }
}

/**
  *   I2CDMAжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CDMAжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Receive_DMA_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴIRCDEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2CDMAж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_IRCDE_POS);
    }
    else
    {
        /* ֹI2CDMAж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_IRCDE_POS);
    }
}

/**
  *   I2CDMAжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2CDMAжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Transmit_DMA_INT_Enable (I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴITXDEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹI2CDMAж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_ITXDE_POS);
    }
    else
    {
        /* ֹI2CDMAж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_ITXDE_POS);
    }
}

/**
  *   I2Cߴжʹܡ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *       NewState: I2Cߴжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
I2C_Bus_Error_INT_Enable(I2C_SFRmap* I2Cx,FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- I2C_IERĴBERRIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹߴж */
        SFR_SET_BIT_ASM(I2Cx->IER, I2C_IER_BERRIE_POS);
    }
    else
    {
        /* δʹߴж */
        SFR_CLR_BIT_ASM(I2Cx->IER, I2C_IER_BERRIE_POS);
    }
}


/**
  *   ȡI2Cʼźű־λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ϳʼλ
  *       0: δʼλ
  */
FlagStatus
I2C_Get_Start_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴSIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_SIF)
    {
        /* ϳʼλ */
        return SET;
    }
    else
    {
        /* δʼλ */
        return RESET;
    }
}

/**
  *   I2Cʼźű־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_Start_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴSIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_SIF)>>I2C_SR_SIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_SIF_POS);
    }
}

/**
  *   ȡI2Cֹͣźű־λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ϳֹͣλ
  *       0: δֹͣλ
  */
FlagStatus
I2C_Get_Stop_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴPIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_PIF)
    {
        /* ϳֹͣλ */
        return SET;
    }
    else
    {
        /* δֹͣλ */
        return RESET;
    }
}

/**
  *   I2Cֹͣźű־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_Stop_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴPIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_PIF)>>I2C_SR_PIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_PIF_POS);
    }
}

/**
  *   ȡI2Cַƥ״̬λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ӻյƥַӦ𣻣ϳֹͣλӲ״̬λ
  *       0: ӻδյӦַ
  */
FlagStatus
I2C_Get_Address_Match_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴADDRλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_ADDR)
    {
        /* ӻյƥַӦ */
        return SET;
    }
    else
    {
        /* ӻδյӦַ */
        return RESET;
    }
}

/**
  *   ȡI2Cλַ״̬λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ϴνջ͵ֽǸλַ
  *       0: ϴνջ͵ֽڲǸλַ
  */
FlagStatus
I2C_Get_HighAddress_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴADD10λ ----------------*/
    if ((I2Cx->SR) & I2C_SR_ADD10)
    {
        /* ϴνջ͵ֽǸλַ */
        return SET;
    }
    else
    {
        /* ϴνջ͵ֽڲǸλַ*/
        return RESET;
    }
}

/**
  *   ȡI2C״̬λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ʾϴνջ͵ֽݣϳʼλ״̬λ
  *       0: ʾϴνջ͵ֽǵַ
  */
FlagStatus
I2C_Get_Data_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴDATAλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_DATA)
    {
        /* ʾϴνջ͵ֽ */
        return SET;
    }
    else
    {
        /* ʾϴνջ͵ֽǵַ*/
        return RESET;
    }
}

/**
  *   ȡI2CӦ־λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: Ӧ
  *       0: δӦ
  */
FlagStatus
I2C_Get_Ack_Fail_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴAFIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_AFIF)
    {
        /* ʾϴνջ͵ֽ */
        return SET;
    }
    else
    {
        /* ʾϴνջ͵ֽǵַ*/
        return RESET;
    }
}

/**
  *   I2CӦ־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_Ack_Fail_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴAFIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_AFIF)>>I2C_SR_AFIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_AFIF_POS);
    }
}

/**
  *   ȡI2Cʧȥٲñ־λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ݹʧȥٲã
  *       0: ݹδʧȥٲá
  */
FlagStatus
I2C_Get_Arbitration_Lost_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴARBLIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_ARBLIF)
    {
        /* ݹʧȥٲ */
        return SET;
    }
    else
    {
        /* ݹδʧȥٲ*/
        return RESET;
    }
}

/**
  *   I2Cʧȥٲñ־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_Arbitration_Lost_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴARBLIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_ARBLIF)>>I2C_SR_ARBLIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_ARBLIF_POS);
    }
}

/**
  *   ȡI2C/ дϢ״̬λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: Ϣ״̬λ
  *       0: дϢ״̬λ
  */
FlagStatus
I2C_Get_Write_Read_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴRNWλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_RNW)
    {
        /* Ϣ״̬λ*/
        return SET;
    }
    else
    {
        /* дϢ״̬λ*/
        return RESET;
    }
}

/**
  *   ȡSMBusжϱ־״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: Ϣ״̬λ
  *       0: дϢ״̬λ
  */
FlagStatus
I2C_Get_SMBus_Alert_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴSMBAIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_SMBAIF)
    {
        /* SMBus¼*/
        return SET;
    }
    else
    {
        /* SMBus*/
        return RESET;
    }
}

/**
  *   SMBusжϱ־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_SMBus_Alert_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴSMBAIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_SMBAIF)>>I2C_SR_SMBAIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_SMBAIF_POS);
    }
}

/**
  *   ȡSMBusͷϵжϱ־״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: Ϣ״̬λ
  *       0: дϢ״̬λ
  */
FlagStatus
I2C_Get_SMBus_Host_Header_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴSMBHIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_SMBHIF)
    {
        /* SMBus¼*/
        return SET;
    }
    else
    {
        /* SMBus*/
        return RESET;
    }
}

/**
  *   SMBusͷϵжϱ־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_SMBus_Host_Header_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴSMBHIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_SMBHIF)>>I2C_SR_SMBHIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_SMBHIF_POS);
    }
}

/**
  *   ȡSMBus豸Ĭϵַ(ģʽ) 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: Ϣ״̬λ
  *       0: дϢ״̬λ
  */
FlagStatus
I2C_Get_SMBus_Device_Default_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴSMBDIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_SMBDIF)
    {
        /* ARPEN=1ʱյSMBus豸Ĭϵַ*/
        return SET;
    }
    else
    {
        /* δյSMBus豸Ĭϵַ*/
        return RESET;
    }
}

/**
  *   SMBus豸Ĭϵַ(ģʽ)־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_SMBus_Device_Default_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴSMBDIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_SMBDIF)>>I2C_SR_SMBDIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_SMBDIF_POS);
    }
}

/**
  *   ȡI2Cжźű־λ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: I2Cжźű־λ
  *       0: δI2Cжźű־λ
  */
FlagStatus
I2C_Get_INTERRUPT_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴISIFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_ISIF)
    {
        /* I2Cжźű־λ*/
        return SET;
    }
    else
    {
        /* δI2Cжźű־λ*/
        return RESET;
    }
}

/**
  *   I2Cжźű־
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   ޡ
  */
void
I2C_Clear_INTERRUPT_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- I2C_SRĴISIFλ ----------------*/
    while(((I2Cx->SR) & I2C_SR_ISIF)>>I2C_SR_ISIF_POS)
    {
    	SFR_CLR_BIT_ASM(I2Cx->SR, I2C_SR_ISIF_POS);
    }
}

/**
  *   ȡI2CBUFFΪ״̬ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: BUFFΪ( I2Cx_BUFR λ)
  *       0: BUFFΪա
  */
FlagStatus
I2C_Get_Receive_Buff_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴRCBFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_RCBF)
    {
        /* BUFFΪ*/
        return SET;
    }
    else
    {
        /* BUFFΪ*/
        return RESET;
    }
}

/**
  *   ȡI2CBUFF״̬λ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: ȴдI2Cx_BUFR(д I2Cx_BUFR λ)
  *       0: ҪдI2Cx_BUFR
  */
FlagStatus
I2C_Get_Transmit_Buff_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴTXBEλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_TXBE)
    {
        /* ȴдI2Cx_BUFR*/
        return SET;
    }
    else
    {
        /* ҪдI2Cx_BUFR*/
        return RESET;
    }
}

/**
  *   ȡI2CDMAжϱ־λ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: I2CDMAжϣ
  *       0: δI2CDMAжϡ
  */
FlagStatus
I2C_Get_Receive_DMA_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴIRCDFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_IRCDF)
    {
        /* I2CDMAж*/
        return SET;
    }
    else
    {
        /* δI2CDMAж*/
        return RESET;
    }
}

/**
  *   ȡI2CDMAжϱ־λ 
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   1: I2CDMAжϣ
  *       0: δI2CDMAжϡ
  */
FlagStatus
I2C_Get_Transmit_DMA_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴITXDFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_ITXDF)
    {
        /* I2CDMAж*/
        return SET;
    }
    else
    {
        /* δI2CDMAж*/
        return RESET;
    }
}

/**
  *   ȡI2Cߴ־λ
  *   I2Cx: ָI2Cڴṹָ룬ȡֵΪI2C0_SFR~I2C3_SFR
  *   	1:  I2C ߴ
  *       	0: δ I2C ߴ
  */
FlagStatus
I2C_Get_Bus_Error_Flag (I2C_SFRmap* I2Cx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_I2C_ALL_PERIPH(I2Cx));

    /*---------------- ȡI2C_SRĴITXDFλ ----------------*/
    if ((I2Cx->SR) & I2C_SR_BERRIF)
    {
        /* I2CDMAж*/
        return SET;
    }
    else
    {
        /* δI2CDMAж*/
        return RESET;
    }
}

/**
  *   ##### ڲɵ·ӿ(I2C)жϹ #####
  */


