/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : kf32a156_CANFD.c                              	      *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This file provides functions related to the CANFD bus *
 *  					bus (CANFD) , including:							  *
 *          			+ CAN Bus(CANFD)initialization function               *
 *          			+ CAN Bus(CANFD)function configuration function       *
 *          			+ CAN Bus(CANFD)transmit-receive function             *
 *          			+ CAN Bus(CANFD)interrupt management function         *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/
/******************************************************************************
**                      		Include Files                                **
******************************************************************************/
#include "kf32a156_canfd.h"
/******************************************************************************
*                      Variables      Declarations                            *
******************************************************************************/
volatile uint8_t a156canfd_txflag = 0x00;
volatile uint8_t arbwinnermailboxnumber = 0x00;


/******************************************************************************
*                      Private Function Declarations                          *
******************************************************************************/
static void Can_m_FdBaudrateSet(const uint8_t Can_Controller_Index, const Can_ControllerConfigType *Can_ControllerConfig, const uint8_t IntIndex);
static void Can_m_FdIntSet(const uint8_t Can_Controller_Index,const Can_ControllerConfigType *Can_ControllerConfig, const uint8_t IntIndex);
static void Can_m_FdFilterInit(const uint8_t Can_Controller_Index,const Can_ControllerConfigType *Can_ControllerConfig,const uint8_t IntIndex);
static Code_Segment Can_m_FdCsCodeSet(Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdWaitConfilct(const uint8_t Can_Controller_Index);
static Can_ReturnType Can_m_FdCopyDataTo8MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataFrom8MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataTo16MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataFrom16MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataTo32MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataFrom32MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataTo64MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);
static Can_ReturnType Can_m_FdCopyDataFrom64MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader);


/******************************************************************************
*                      Private Function Declarations                          *
******************************************************************************/

const Can_ControllerInfoType Can_m_ControllersInfo[3] = {
	{CAN_HW_CONTROLLER_BASEADDRESS_CAN4,
	 CAN_HW_CONTROLLER_MBADDRESS_CAN4,
	 CAN_HW_CONTROLLER_FILTERADDRESS_CAN4,
	 CAN_CONTROLLER_CLASSICAL},

	{CAN_HW_CONTROLLER_BASEADDRESS_CANFD6,
	 CAN_HW_CONTROLLER_MBADDRESS_CANFD6,
	 CAN_HW_CONTROLLER_FILTERADDRESS_CANFD6,
	 CAN_CONTROLLER_FD},

	{CAN_HW_CONTROLLER_BASEADDRESS_CANFD7,
	 CAN_HW_CONTROLLER_MBADDRESS_CANFD7,
	 CAN_HW_CONTROLLER_FILTERADDRESS_CANFD7,
	 CAN_CONTROLLER_FD}
};


/* Set CAN Default baudrate */
static void Can_m_FdBaudrateSet(const uint8_t Can_Controller_Index, const Can_ControllerConfigType *Can_ControllerConfig, const uint8_t IntIndex)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	/* Set CAN Default baudrate */
	/* [$Block Start$] */
	/* Set Can Controller Sample time */
	ControllerRegPtr->CANFD_BRGR.B.SAM = Can_ControllerConfig[IntIndex].Canfd_Controller_AllClockAndBDRConfig->SampleTimes;
	/* Set Can Controller Synchronization Jump Width */
	ControllerRegPtr->CANFD_BRGR.B.SJW = Can_ControllerConfig[IntIndex].Canfd_Controller_AllClockAndBDRConfig->Sjw;
	/* Set Can Controller Baud Rate Clock Prescale */
	ControllerRegPtr->CANFD_BRGR.B.CNABRP = Can_ControllerConfig[IntIndex].Canfd_Controller_AllClockAndBDRConfig->PreScale;
	/* Set T Segment 1 */
	ControllerRegPtr->CANFD_BRGR.B.TSEG1 = Can_ControllerConfig[IntIndex].Canfd_Controller_AllClockAndBDRConfig->TSeg1;
	/* Set T Segment 2 */
	ControllerRegPtr->CANFD_BRGR.B.TSEG2 = Can_ControllerConfig[IntIndex].Canfd_Controller_AllClockAndBDRConfig->TSeg2;
	/* [$Block End$] */


	/* [$Block Start$] */
	/* Set Can Controller Baud Rate Switch Prescale */
	ControllerRegPtr->CANFD_BRGR.B.BRSBRP = Can_ControllerConfig[IntIndex].Canfd_Controller_ALLFdBDRConfig->BrsPrescale;
	/* Set Can Controller Baud Rate Switch T Segment 1 */
	ControllerRegPtr->CANFD_BRGR.B.HTSEG1 = Can_ControllerConfig[IntIndex].Canfd_Controller_ALLFdBDRConfig->HtSeg1;
	/* Set Can Controller Baud Rate Switch T Segment 2*/
	ControllerRegPtr->CANFD_BRGR.B.HTSEG2 = Can_ControllerConfig[IntIndex].Canfd_Controller_ALLFdBDRConfig->HtSeg2;
	/* [$Block End$] */
}


/**
  *   CANFD
  *   Can_Controller_Index: CANFDţȡֵΪCANfd6CANfd7
  *   ޡ
  */
void Can_m_FdMailBoxErase(const uint8_t Can_Controller_Index)
{
	for (uint8_t erase_count = 0; erase_count < 205; erase_count++)
	{
		*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + erase_count) = 0x00000000;
	}
}


/**
  *   CANFD
  *   Can_Controller_Index: CANFDţȡֵΪCANfd6CANfd7
  *   ޡ
  */
void Can_m_FdMailBoxMaskErase(const uint8_t Can_Controller_Index)
{
	for (uint16_t erase_count = 205; erase_count < 256; erase_count++)
	{
		*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + erase_count) = 0x00000000;
	}
}


/**
  *   CANFDʼ
  *    Can_Controller_Index: CANFDţȡֵΪCANfd6CANfd7
  * 	 Can_ControllerConfigCANFDͽṹ塣
  * 	 InitIndexţȡֵΪInitindex_0~Initindex_4
  *   ޡ
  */
void Can_m_FdControllerInit(const uint8_t Can_Controller_Index,const Can_ControllerConfigType *Can_ControllerConfig, const Can_InitIndexType InitIndex)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	volatile uint16_t delay_count = 0xff;
	/* Disable CAN controller */
	ControllerRegPtr->CANFD_CTLR0.B.CANEN = STD_OFF;
	/* Set CAN controller to reset work mode */
	ControllerRegPtr->CANFD_CTLR0.B.RSMOD = CAN_CONTROLLER_RUN_MODE_RESET;
	/* Waitting For Can Controller Go to reset mode */
	while(delay_count--)/*  compound statement  */
	{

	};
	/* Disable Listen-only mode */
	ControllerRegPtr->CANFD_CTLR0.B.SILENT = CAN_CONTROLLER_MODE_DISABLE_LISTENONLY;
	/* Disable sleep mode */
	ControllerRegPtr->CANFD_CTLR0.B.SLEEP = CAN_CONTROLLER_MODE_DISABLE_SLEEP;
	/* Set Canfd To Loop Mode */
	if (Can_ControllerConfig[InitIndex].CanfdMode == CANFD_LOOP_INTERNAL_MODE)
	{
		ControllerRegPtr->CANFD_CTLR0.B.LBACK = CAN_CONTROLLER_MODE_ENABLE_LOOP;
		ControllerRegPtr->CANFD_CTLR1.B.LBSEL = CAN_CONTROLLER_MODE_ENABLE_INTERLOOP;
	}else if(Can_ControllerConfig[InitIndex].CanfdMode == CNAFD_LOOP_EXTERNAL_MODE)
	{
		ControllerRegPtr->CANFD_CTLR0.B.LBACK = CAN_CONTROLLER_MODE_ENABLE_LOOP;
		ControllerRegPtr->CANFD_CTLR1.B.LBSEL = CAN_CONTROLLER_MODE_DISABLE_INTERLOOP;
	}else if(Can_ControllerConfig[InitIndex].CanfdMode == CANFD_SILENT_MODE)
	{
		ControllerRegPtr->CANFD_CTLR0.B.LBACK = CAN_CONTROLLER_MODE_DISABLE_INTERLOOP;
		ControllerRegPtr->CANFD_CTLR0.B.SILENT = CAN_CONTROLLER_MODE_ENABLE_LISTENONLY;
	}else if(Can_ControllerConfig[InitIndex].CanfdMode == CANFD_NORMAL_MODE)
	{
		ControllerRegPtr->CANFD_CTLR0.B.LBACK = CAN_CONTROLLER_MODE_DISABLE_INTERLOOP;
		ControllerRegPtr->CANFD_CTLR0.B.SILENT = CAN_CONTROLLER_MODE_DISABLE_LISTENONLY;
	}

#if (WORKSOURCE_DIVIDE_1 == STD_ON)
	ControllerRegPtr->CANFD_CTLR1.B.CKMODE = STD_ON;
#else
	ControllerRegPtr->CANFD_CTLR1.B.CKMODE = STD_OFF;
#endif

#if (TRANSMIT_DELAY_CONPENSATION == STD_ON)
#if (TRANSMIT_DELAY_CONPENSATION_HARDWARE == STD_ON)
	/* Enable Transmiter Delay Conpensation */
	ControllerRegPtr->CANFD_CTLR2.B.TDCEN = 0x01;
	ControllerRegPtr->CANFD_CTLR2.B.TDCOEN = 0x01;
	ControllerRegPtr->CANFD_CTLR2.B.TDCSAUTO = 0x01;

#else
	ControllerRegPtr->CANFD_CTLR2.B.TDCEN = 0x01;
	ControllerRegPtr->CANFD_CTLR2.B.TDCOEN = 0x01;
	ControllerRegPtr->CANFD_CTLR2.B.TDCO = 0x03;
	ControllerRegPtr->CANFD_CTLR2.B.TDCS = 0x03;
#endif

#endif

	/* Set Arbitrate Clock Divide */
	ControllerRegPtr->CANFD_CTLR2.B.AMCKDIV = Clock_Divide_1;
	/* Set Canfd Frame type */
	ControllerRegPtr->CANFD_CTLR0.B.ISOFDCAN = Can_ControllerConfig[InitIndex].FdFrameType;
	/* Set Canfd MailBox Block Size */
	ControllerRegPtr->CANFD_CTLR0.B.MBSIZE = Can_ControllerConfig[InitIndex].MailBoxBlockSize;
	/* Set Whether Full Canfd MailBox Participate the Receive Match*/
	ControllerRegPtr->CANFD_CTLR0.B.FULLRXEN = Can_ControllerConfig[InitIndex].MBFullReceiveEnableSet;
	/* Set Can Controller Work Clock Source */
	ControllerRegPtr->CANFD_CTLR0.B.CANCKS = Can_ControllerConfig[InitIndex].ClockSource;
	/* Set Can Controller Arbitrate Clock Source */
	ControllerRegPtr->CANFD_CTLR1.B.AMCKS = Can_ControllerConfig[InitIndex].FdArbitrateClockSource;
	/* Set Can Controller Global Mask Code */
	ControllerRegPtr->CANFD_MSKR = Can_ControllerConfig[InitIndex].GlobalMask;

	ControllerRegPtr->CANFD_CTLR1.B.BOFFREC = 0x01;

	/* Enable CANFD Mode */
	ControllerRegPtr->CANFD_CTLR0.B.CANFDEN = 0x01;
	/* Set CAN Default baudrate */
	Can_m_FdBaudrateSet(Can_Controller_Index, Can_ControllerConfig,0);
	/* Set The Can Intterrupt  Enable And Clear The Interrupt Flag*/
	Can_m_FdIntSet(Can_Controller_Index, Can_ControllerConfig,0);
	/* Set The Can Filter Code */
	Can_m_FdFilterInit(Can_Controller_Index,Can_ControllerConfig,0);

	/* Enable Interrupt Module */
	if (Can_Controller_Index == 0)
	{
		INTERRUPT_REG.EIE1.B.CAN4IE = 0x01;
	}
	else if (Can_Controller_Index == 1)
	{
		INTERRUPT_REG.EIE2.B.CANFD6IE = 0x01;
	}
	else if (Can_Controller_Index == 2)
	{
		INTERRUPT_REG.EIE2.B.CANFD7IE = 0x01;
	}
	else
	{
	}
	INTERRUPT_REG.CTL0.B.AIE = 0x01;

	/* Enable The Can Controller Module */
	ControllerRegPtr->CANFD_CTLR0.B.CANEN = STD_ON;
	/* Set CAN Controller to work mode */
	ControllerRegPtr->CANFD_CTLR0.B.RSMOD = CAN_CONTROLLER_RUN_MODE_RUN;
	/* CAN  Controller Interrupt Settings */
	while(delay_count--);
}



/**
  *   CANFDλ
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *   ޡ
  */
void Can_m_FdControllerDeInit(const uint8_t Can_Controller_Index)
{
	if (Can_m_ControllersInfo[Can_Controller_Index].BaseAddress == CAN_HW_CONTROLLER_BASEADDRESS_CAN4)
	{
		RST.CTL3.B.CAN4RST = 0x01;
		PCLK.CTL3.B.CAN4CLKEN = 0x01;
		RST.CTL3.B.CAN4RST = 0x00;
	}
	else if (Can_m_ControllersInfo[Can_Controller_Index].BaseAddress == CAN_HW_CONTROLLER_BASEADDRESS_CANFD6)
	{
		Can_m_FdMailBoxErase(Can_Controller_Index);
		RST.CTL3.B.CNAFD6RST = 0x01;
		PCLK.CTL3.B.CANFD6CLKEN = 0x01;
		RST.CTL3.B.CNAFD6RST = 0x00;
	}
	else if (Can_m_ControllersInfo[Can_Controller_Index].BaseAddress == CAN_HW_CONTROLLER_BASEADDRESS_CANFD7)
	{
		Can_m_FdMailBoxErase(Can_Controller_Index);
		RST.CTL3.B.CNAFD7RST = 0x01;
		PCLK.CTL3.B.CANFD7CLKEN = 0x01;
		RST.CTL3.B.CNAFD7RST = 0x00;
	}
	else
	{
	}
}



/* Set The Can Intterrupt  Enable And Clear The Interrupt Flag*/
static void Can_m_FdIntSet(const uint8_t Can_Controller_Index,const Can_ControllerConfigType *Can_ControllerConfig, const uint8_t IntIndex)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	/* Clear All Interrupt Flags */
	ControllerRegPtr->CANFD_IFR.R &= ~0x7FFF;
	ControllerRegPtr->CANFD_IER.B.TRANSMITIC = 0x01;
	/* CAN Receive Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.CANRXIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.RxIntEnableSet;
	/* CAN Transmit Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.CANTXIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.TxIntEnableSet;
	/* CAN Bus Off Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.BOFFIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.BusOffEnableSet;
	/* CAN Wake Up Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.WUIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.WakeUpIntEnableSet;
	/* CAN Error Alarm Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.EAIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.ErrorAlarmIntEnableSet;
	/* CAN Receive Over Flow Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.DOVFIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.OverFlowIntEnableSet;
	/* CAN Error Negative Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.ENIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.ErrorNegativeIntEnableSet;
	/* CAN Error Arbitrate Lose Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.ALIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.ArbitrateLoseIntEnableSet;
	/* CAN Bus Error Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.BEIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.BusErrorIntEnableSet;
	/* CAN DMA Transmit Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.CTXDE = Can_ControllerConfig[IntIndex].InterruptEnableSet.DmaTXDEnableSet;
	/* CAN DMA Receive Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.CRXDE = Can_ControllerConfig[IntIndex].InterruptEnableSet.DmaRXDEnableSet;
	/* CAN Receive Not Empty Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.RXBSTAIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.RXBStateEnableSet;
	/* CAN MailBox Receive Triger Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.TRGMBIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.TrigerMBEnableSet;
	/* CAN Transmit Delay Offset Fail Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.TDCFAILIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.TransmitDelayOffsetFailEnableSet;
	/* CAN Transmit Arbitrate Fail Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.ARBFAILIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.ArbitrateFailEnableSet;
	/* CAN Move Out End Interrupt Set */
	ControllerRegPtr->CANFD_IER.B.MOENDIE = Can_ControllerConfig[IntIndex].InterruptEnableSet.MoveOutEndEnableSet;
}



/* Set The Can Filter Code And Mask Code) */
static void Can_m_FdFilterInit(const uint8_t Can_Controller_Index,const Can_ControllerConfigType *Can_ControllerConfig,const uint8_t IntIndex)
{
	uint8_t MailBoxCount = 0;
	Can_m_FdMailBoxMaskErase(Can_Controller_Index);
	/* Configuration settings Addresses */
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	/* Enable MailBox Mask */
	ControllerRegPtr->CANFD_CTLR1.B.MBMSKEN = STD_ON;

	if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_8MAILBOX_NUMBER; mailbox_count++)
		{
			if(Can_ControllerConfig[IntIndex].Can_HwFilter[0].Can_IdFrame[mailbox_count] == CAN_ID_STANDARD)
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<21);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x1FFFFF;
			}else
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<3);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x07;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_16MAILBOX_NUMBER; mailbox_count++)
		{
			if(Can_ControllerConfig[IntIndex].Can_HwFilter[0].Can_IdFrame[mailbox_count] == CAN_ID_STANDARD)
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<21);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x1FFFFF;
			}else
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<3);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x07;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_32MAILBOX_NUMBER; mailbox_count++)
		{
			if(Can_ControllerConfig[IntIndex].Can_HwFilter[0].Can_IdFrame[mailbox_count] == CAN_ID_STANDARD)
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<21);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x1FFFFF;
			}else
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<3);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x07;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_64MAILBOX_NUMBER; mailbox_count++)
		{
			if(Can_ControllerConfig[IntIndex].Can_HwFilter[0].Can_IdFrame[mailbox_count] == CAN_ID_STANDARD)
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<21);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x1FFFFF;
			}else
			{
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) = ((Can_ControllerConfig[IntIndex].Can_HwFilter[0].MaskCode[mailbox_count])<<3);
				*(((uint32_t *)Can_m_ControllersInfo[Can_Controller_Index].FilterBaseAddress) - mailbox_count) |= 0x07;
			}
		}
	}
	else
	{
	}
}


/* Make The MailBox Code Segment Configuration Into Integer Number*/
static Code_Segment Can_m_FdCsCodeSet(Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	Code_Segment Ret_Code_Seg = {0};
	if(Can_MailboxHeader->BRS == 0x01)
	{
		Ret_Code_Seg.BRS = 0x01;
	}

	if(Can_MailboxHeader->Can_id == CAN_ID_EXTENDED)
	{
		Ret_Code_Seg.IDE = 0x01;
	}

	if(Can_MailboxHeader->Can_frame == CAN_FRAME_FD)
	{
		Ret_Code_Seg.EDL = 0x01;
	}

	Ret_Code_Seg.DLC = Can_MailboxHeader->FrameData.Data_Length;
	Ret_Code_Seg.CODE = Can_MailboxHeader->TransceiveType;

	if(Can_MailboxHeader->TransceiveType == MAIL_RTR_REQUEST)
	{
		Ret_Code_Seg.CODE = 0x0C;
		Ret_Code_Seg.RTR = 0x01;
	}else
	{
		Ret_Code_Seg.RTR = 0x00;
	}
	return Ret_Code_Seg;
}


/* Avoid Cpu And Hardware Can Module Conflict */
static Can_ReturnType Can_m_FdWaitConfilct(const uint8_t Can_Controller_Index)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	while(1)
	{
		if(ControllerRegPtr->CANFD_CTLR0.B.TXSTA == 0x01)
		{
			if((ControllerRegPtr->CANFD_CTLR0.B.TCSTA == 0x00)&&(ControllerRegPtr->CANFD_AMSTA.B.ARBSTA0 == 0x00)&&(ControllerRegPtr->CANFD_IFR.B.MOENDIF == 0x00))
			{
				ret = CAN_OK;
				break;
			}

			if((ControllerRegPtr->CANFD_IFR.B.MOENDIF == 0x01) && ((ControllerRegPtr->CANFD_AMSTA.B.ARBSTA0 == 0x01)||(ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x01)))
			{
				ControllerRegPtr->CANFD_IER.B.MOENDIC = 0x01;
				while (ControllerRegPtr->CANFD_IFR.B.MOENDIF == 0x01)
					;
				ControllerRegPtr->CANFD_IER.B.MOENDIC = 0x00;
				ret = CAN_OK;
				break;
			}
		}
		else if(ControllerRegPtr ->CANFD_CTLR0.B.RXSTA == 0x01)
		{
			if(ControllerRegPtr->CANFD_AMSTA.B.MATSTA == 0x01)
			{
				ret = CAN_OK;
				break;
			}
		}else
		{
			ret = CAN_OK;
			break;
		}
	}
	return ret;
}


/* Set Configuration Infor To Ram */
static Can_ReturnType Can_m_FdCopyDataTo8MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	Can_IdFrameType can_id = Can_MailboxHeader->Can_id;
	Code_Segment Ret_Code_Segment = {0};
	Ret_Code_Segment = Can_m_FdCsCodeSet(Can_MailboxHeader);
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if(can_id == CAN_ID_STANDARD)
		{
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];
		}
		else if (can_id == CAN_ID_EXTENDED)
		{
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];
		}
		((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
	}else
	{

	}
	return ret;
}




/* Get Message Infor From Ram */
static Can_ReturnType Can_m_FdCopyDataFrom8MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.IDE == 0x00)
		{
			Can_MailboxHeader->Can_id = CAN_ID_STANDARD;
		}
		else
		{
			Can_MailboxHeader->Can_id = CAN_ID_EXTENDED;
		}

		if (((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.EDL == 0x00)
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_CLASSICAL;
		}
		else
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_FD;
		}

		if (Can_MailboxHeader->Can_id == CAN_ID_STANDARD)
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID;
			Can_MailboxHeader->BRS = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.BRS;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DLC;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7;
		}
		else
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID;
			Can_MailboxHeader->BRS = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.BRS;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DLC;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7;
		}
	}else
	{

	}
	return ret;
}


/* Set Configuration Infor To Ram */
static Can_ReturnType Can_m_FdCopyDataTo16MailBox(const uint8_t Can_Controller_Index,const Can_MailboxNumType MailBox_Number,Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_IdFrameType can_id = Can_MailboxHeader->Can_id;
	Code_Segment Ret_Code_Segment = {0};
	Can_ReturnType ret = CAN_BUSY;
	Ret_Code_Segment =  Can_m_FdCsCodeSet(Can_MailboxHeader);
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (can_id == CAN_ID_STANDARD)
		{
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];
	}
	else if (can_id == CAN_ID_EXTENDED)
	{
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];
		}
		((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
	}else
	{

	}
	return ret;
}


/* Get Message Infor From Ram */
static Can_ReturnType Can_m_FdCopyDataFrom16MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.IDE == 0x00)
		{
			Can_MailboxHeader->Can_id = CAN_ID_STANDARD;
		}
		else
		{
			Can_MailboxHeader->Can_id = CAN_ID_EXTENDED;
		}

		if (((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.EDL == 0x00)
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_CLASSICAL;
		}
		else
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_FD;
		}

		if (Can_MailboxHeader->Can_id == CAN_ID_STANDARD)
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15;
		}
		else
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15;
		}
	}else
	{

	}
	return ret;
}


/* Set Configuration Infor To Ram */
static Can_ReturnType Can_m_FdCopyDataTo32MailBox(const uint8_t Can_Controller_Index,const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	Can_IdFrameType can_id = Can_MailboxHeader->Can_id;
	Code_Segment Ret_Code_Segment = {0};
	Ret_Code_Segment =  Can_m_FdCsCodeSet(Can_MailboxHeader);
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (can_id == CAN_ID_STANDARD)
		{
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA16 = Can_MailboxHeader->FrameData.U8Data[16];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA17 = Can_MailboxHeader->FrameData.U8Data[17];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA18 = Can_MailboxHeader->FrameData.U8Data[18];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA19 = Can_MailboxHeader->FrameData.U8Data[19];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA20 = Can_MailboxHeader->FrameData.U8Data[20];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA21 = Can_MailboxHeader->FrameData.U8Data[21];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA22 = Can_MailboxHeader->FrameData.U8Data[22];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA23 = Can_MailboxHeader->FrameData.U8Data[23];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA24 = Can_MailboxHeader->FrameData.U8Data[24];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA25 = Can_MailboxHeader->FrameData.U8Data[25];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA26 = Can_MailboxHeader->FrameData.U8Data[26];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA27 = Can_MailboxHeader->FrameData.U8Data[27];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA28 = Can_MailboxHeader->FrameData.U8Data[28];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA29 = Can_MailboxHeader->FrameData.U8Data[29];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA30 = Can_MailboxHeader->FrameData.U8Data[30];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA31 = Can_MailboxHeader->FrameData.U8Data[31];
		}
		else if (can_id == CAN_ID_EXTENDED)
		{
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA16 = Can_MailboxHeader->FrameData.U8Data[16];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA17 = Can_MailboxHeader->FrameData.U8Data[17];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA18 = Can_MailboxHeader->FrameData.U8Data[18];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA19 = Can_MailboxHeader->FrameData.U8Data[19];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA20 = Can_MailboxHeader->FrameData.U8Data[20];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA21 = Can_MailboxHeader->FrameData.U8Data[21];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA22 = Can_MailboxHeader->FrameData.U8Data[22];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA23 = Can_MailboxHeader->FrameData.U8Data[23];

			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA24 = Can_MailboxHeader->FrameData.U8Data[24];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA25 = Can_MailboxHeader->FrameData.U8Data[25];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA26 = Can_MailboxHeader->FrameData.U8Data[26];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA27 = Can_MailboxHeader->FrameData.U8Data[27];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA28 = Can_MailboxHeader->FrameData.U8Data[28];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA29 = Can_MailboxHeader->FrameData.U8Data[29];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA30 = Can_MailboxHeader->FrameData.U8Data[30];
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA31 = Can_MailboxHeader->FrameData.U8Data[31];
		}
		((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
	}else
	{

	}
	return ret;
}



/* Get Configuration Infor From Ram */
static Can_ReturnType Can_m_FdCopyDataFrom32MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.IDE == 0x00)
		{
			Can_MailboxHeader->Can_id = CAN_ID_STANDARD;
		}
		else
		{
			Can_MailboxHeader->Can_id = CAN_ID_EXTENDED;
		}

		if (((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.EDL == 0x00)
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_CLASSICAL;
		}
		else
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_FD;
		}

		if (Can_MailboxHeader->Can_id == CAN_ID_STANDARD)
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15;

			Can_MailboxHeader->FrameData.U8Data[16] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA16;
			Can_MailboxHeader->FrameData.U8Data[17] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA17;
			Can_MailboxHeader->FrameData.U8Data[18] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA18;
			Can_MailboxHeader->FrameData.U8Data[19] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA19;
			Can_MailboxHeader->FrameData.U8Data[20] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA20;
			Can_MailboxHeader->FrameData.U8Data[21] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA21;
			Can_MailboxHeader->FrameData.U8Data[22] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA22;
			Can_MailboxHeader->FrameData.U8Data[23] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA23;

			Can_MailboxHeader->FrameData.U8Data[24] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA24;
			Can_MailboxHeader->FrameData.U8Data[25] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA25;
			Can_MailboxHeader->FrameData.U8Data[26] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA26;
			Can_MailboxHeader->FrameData.U8Data[27] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA27;
			Can_MailboxHeader->FrameData.U8Data[28] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA28;
			Can_MailboxHeader->FrameData.U8Data[29] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA29;
			Can_MailboxHeader->FrameData.U8Data[30] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA30;
			Can_MailboxHeader->FrameData.U8Data[31] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA31;
		}
		else
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15;

			Can_MailboxHeader->FrameData.U8Data[16] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA16;
			Can_MailboxHeader->FrameData.U8Data[17] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA17;
			Can_MailboxHeader->FrameData.U8Data[18] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA18;
			Can_MailboxHeader->FrameData.U8Data[19] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA19;
			Can_MailboxHeader->FrameData.U8Data[20] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA20;
			Can_MailboxHeader->FrameData.U8Data[21] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA21;
			Can_MailboxHeader->FrameData.U8Data[22] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA22;
			Can_MailboxHeader->FrameData.U8Data[23] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA23;

			Can_MailboxHeader->FrameData.U8Data[24] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA24;
			Can_MailboxHeader->FrameData.U8Data[25] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA25;
			Can_MailboxHeader->FrameData.U8Data[26] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA26;
			Can_MailboxHeader->FrameData.U8Data[27] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA27;
			Can_MailboxHeader->FrameData.U8Data[28] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA28;
			Can_MailboxHeader->FrameData.U8Data[29] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA29;
			Can_MailboxHeader->FrameData.U8Data[30] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA30;
			Can_MailboxHeader->FrameData.U8Data[31] = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA31;
		}
	}else
	{

	}
	return ret;
}


/* Set Configuration Infor To Ram */
static Can_ReturnType Can_m_FdCopyDataTo64MailBox(const uint8_t Can_Controller_Index,const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	Can_IdFrameType can_id = Can_MailboxHeader->Can_id;
	Code_Segment Ret_Code_Segment = {0};
	Ret_Code_Segment =  Can_m_FdCsCodeSet(Can_MailboxHeader);
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (can_id == CAN_ID_STANDARD)
		{
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA16 = Can_MailboxHeader->FrameData.U8Data[16];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA17 = Can_MailboxHeader->FrameData.U8Data[17];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA18 = Can_MailboxHeader->FrameData.U8Data[18];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA19 = Can_MailboxHeader->FrameData.U8Data[19];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA20 = Can_MailboxHeader->FrameData.U8Data[20];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA21 = Can_MailboxHeader->FrameData.U8Data[21];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA22 = Can_MailboxHeader->FrameData.U8Data[22];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA23 = Can_MailboxHeader->FrameData.U8Data[23];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA24 = Can_MailboxHeader->FrameData.U8Data[24];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA25 = Can_MailboxHeader->FrameData.U8Data[25];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA26 = Can_MailboxHeader->FrameData.U8Data[26];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA27 = Can_MailboxHeader->FrameData.U8Data[27];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA28 = Can_MailboxHeader->FrameData.U8Data[28];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA29 = Can_MailboxHeader->FrameData.U8Data[29];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA30 = Can_MailboxHeader->FrameData.U8Data[30];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA31 = Can_MailboxHeader->FrameData.U8Data[31];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA32 = Can_MailboxHeader->FrameData.U8Data[32];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA33 = Can_MailboxHeader->FrameData.U8Data[33];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA34 = Can_MailboxHeader->FrameData.U8Data[34];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA35 = Can_MailboxHeader->FrameData.U8Data[35];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA36 = Can_MailboxHeader->FrameData.U8Data[36];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA37 = Can_MailboxHeader->FrameData.U8Data[37];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA38 = Can_MailboxHeader->FrameData.U8Data[38];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA39 = Can_MailboxHeader->FrameData.U8Data[39];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA40 = Can_MailboxHeader->FrameData.U8Data[40];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA41 = Can_MailboxHeader->FrameData.U8Data[41];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA42 = Can_MailboxHeader->FrameData.U8Data[42];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA43 = Can_MailboxHeader->FrameData.U8Data[43];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA44 = Can_MailboxHeader->FrameData.U8Data[44];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA45 = Can_MailboxHeader->FrameData.U8Data[45];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA46 = Can_MailboxHeader->FrameData.U8Data[46];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA47 = Can_MailboxHeader->FrameData.U8Data[47];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA48 = Can_MailboxHeader->FrameData.U8Data[48];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA49 = Can_MailboxHeader->FrameData.U8Data[49];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA50 = Can_MailboxHeader->FrameData.U8Data[50];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA51 = Can_MailboxHeader->FrameData.U8Data[51];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA52 = Can_MailboxHeader->FrameData.U8Data[52];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA53 = Can_MailboxHeader->FrameData.U8Data[53];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA54 = Can_MailboxHeader->FrameData.U8Data[54];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA55 = Can_MailboxHeader->FrameData.U8Data[55];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA56 = Can_MailboxHeader->FrameData.U8Data[56];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA57 = Can_MailboxHeader->FrameData.U8Data[57];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA58 = Can_MailboxHeader->FrameData.U8Data[58];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA59 = Can_MailboxHeader->FrameData.U8Data[59];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA60 = Can_MailboxHeader->FrameData.U8Data[60];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA61 = Can_MailboxHeader->FrameData.U8Data[61];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA62 = Can_MailboxHeader->FrameData.U8Data[62];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA63 = Can_MailboxHeader->FrameData.U8Data[63];
		}
		else if (can_id == CAN_ID_EXTENDED)
		{
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID = Can_MailboxHeader->Id;

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0 = Can_MailboxHeader->FrameData.U8Data[0];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1 = Can_MailboxHeader->FrameData.U8Data[1];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2 = Can_MailboxHeader->FrameData.U8Data[2];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3 = Can_MailboxHeader->FrameData.U8Data[3];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4 = Can_MailboxHeader->FrameData.U8Data[4];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5 = Can_MailboxHeader->FrameData.U8Data[5];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6 = Can_MailboxHeader->FrameData.U8Data[6];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7 = Can_MailboxHeader->FrameData.U8Data[7];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8 = Can_MailboxHeader->FrameData.U8Data[8];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9 = Can_MailboxHeader->FrameData.U8Data[9];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10 = Can_MailboxHeader->FrameData.U8Data[10];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11 = Can_MailboxHeader->FrameData.U8Data[11];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12 = Can_MailboxHeader->FrameData.U8Data[12];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13 = Can_MailboxHeader->FrameData.U8Data[13];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14 = Can_MailboxHeader->FrameData.U8Data[14];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15 = Can_MailboxHeader->FrameData.U8Data[15];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA16 = Can_MailboxHeader->FrameData.U8Data[16];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA17 = Can_MailboxHeader->FrameData.U8Data[17];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA18 = Can_MailboxHeader->FrameData.U8Data[18];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA19 = Can_MailboxHeader->FrameData.U8Data[19];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA20 = Can_MailboxHeader->FrameData.U8Data[20];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA21 = Can_MailboxHeader->FrameData.U8Data[21];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA22 = Can_MailboxHeader->FrameData.U8Data[22];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA23 = Can_MailboxHeader->FrameData.U8Data[23];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA24 = Can_MailboxHeader->FrameData.U8Data[24];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA25 = Can_MailboxHeader->FrameData.U8Data[25];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA26 = Can_MailboxHeader->FrameData.U8Data[26];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA27 = Can_MailboxHeader->FrameData.U8Data[27];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA28 = Can_MailboxHeader->FrameData.U8Data[28];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA29 = Can_MailboxHeader->FrameData.U8Data[29];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA30 = Can_MailboxHeader->FrameData.U8Data[30];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA31 = Can_MailboxHeader->FrameData.U8Data[31];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA32 = Can_MailboxHeader->FrameData.U8Data[32];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA33 = Can_MailboxHeader->FrameData.U8Data[33];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA34 = Can_MailboxHeader->FrameData.U8Data[34];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA35 = Can_MailboxHeader->FrameData.U8Data[35];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA36 = Can_MailboxHeader->FrameData.U8Data[36];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA37 = Can_MailboxHeader->FrameData.U8Data[37];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA38 = Can_MailboxHeader->FrameData.U8Data[38];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA39 = Can_MailboxHeader->FrameData.U8Data[39];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA40 = Can_MailboxHeader->FrameData.U8Data[40];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA41 = Can_MailboxHeader->FrameData.U8Data[41];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA42 = Can_MailboxHeader->FrameData.U8Data[42];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA43 = Can_MailboxHeader->FrameData.U8Data[43];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA44 = Can_MailboxHeader->FrameData.U8Data[44];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA45 = Can_MailboxHeader->FrameData.U8Data[45];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA46 = Can_MailboxHeader->FrameData.U8Data[46];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA47 = Can_MailboxHeader->FrameData.U8Data[47];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA48 = Can_MailboxHeader->FrameData.U8Data[48];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA49 = Can_MailboxHeader->FrameData.U8Data[49];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA50 = Can_MailboxHeader->FrameData.U8Data[50];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA51 = Can_MailboxHeader->FrameData.U8Data[51];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA52 = Can_MailboxHeader->FrameData.U8Data[52];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA53 = Can_MailboxHeader->FrameData.U8Data[53];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA54 = Can_MailboxHeader->FrameData.U8Data[54];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA55 = Can_MailboxHeader->FrameData.U8Data[55];

			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA56 = Can_MailboxHeader->FrameData.U8Data[56];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA57 = Can_MailboxHeader->FrameData.U8Data[57];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA58 = Can_MailboxHeader->FrameData.U8Data[58];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA59 = Can_MailboxHeader->FrameData.U8Data[59];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA60 = Can_MailboxHeader->FrameData.U8Data[60];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA61 = Can_MailboxHeader->FrameData.U8Data[61];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA62 = Can_MailboxHeader->FrameData.U8Data[62];
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA63 = Can_MailboxHeader->FrameData.U8Data[63];
		}
		((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
	}else
	{

	}
	return ret;
}



/* Get Configuration Infor From Ram */
static Can_ReturnType Can_m_FdCopyDataFrom64MailBox(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_ReturnType ret = CAN_BUSY;
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if (((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.IDE == 0x00)
		{
			Can_MailboxHeader->Can_id = CAN_ID_STANDARD;
		}
		else
		{
			Can_MailboxHeader->Can_id = CAN_ID_EXTENDED;
		}

		if (((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.EDL == 0x00)
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_CLASSICAL;
		}
		else
		{
			Can_MailboxHeader->Can_frame = CAN_FRAME_FD;
		}

		if (Can_MailboxHeader->Can_id == CAN_ID_STANDARD)
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA15;

			Can_MailboxHeader->FrameData.U8Data[16] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA16;
			Can_MailboxHeader->FrameData.U8Data[17] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA17;
			Can_MailboxHeader->FrameData.U8Data[18] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA18;
			Can_MailboxHeader->FrameData.U8Data[19] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA19;
			Can_MailboxHeader->FrameData.U8Data[20] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA20;
			Can_MailboxHeader->FrameData.U8Data[21] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA21;
			Can_MailboxHeader->FrameData.U8Data[22] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA22;
			Can_MailboxHeader->FrameData.U8Data[23] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA23;

			Can_MailboxHeader->FrameData.U8Data[24] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA24;
			Can_MailboxHeader->FrameData.U8Data[25] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA25;
			Can_MailboxHeader->FrameData.U8Data[26] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA26;
			Can_MailboxHeader->FrameData.U8Data[27] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA27;
			Can_MailboxHeader->FrameData.U8Data[28] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA28;
			Can_MailboxHeader->FrameData.U8Data[29] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA29;
			Can_MailboxHeader->FrameData.U8Data[30] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA30;
			Can_MailboxHeader->FrameData.U8Data[31] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA31;

			Can_MailboxHeader->FrameData.U8Data[32] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA32;
			Can_MailboxHeader->FrameData.U8Data[33] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA33;
			Can_MailboxHeader->FrameData.U8Data[34] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA34;
			Can_MailboxHeader->FrameData.U8Data[35] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA35;
			Can_MailboxHeader->FrameData.U8Data[36] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA36;
			Can_MailboxHeader->FrameData.U8Data[37] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA37;
			Can_MailboxHeader->FrameData.U8Data[38] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA38;
			Can_MailboxHeader->FrameData.U8Data[39] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA39;

			Can_MailboxHeader->FrameData.U8Data[40] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA40;
			Can_MailboxHeader->FrameData.U8Data[41] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA41;
			Can_MailboxHeader->FrameData.U8Data[42] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA42;
			Can_MailboxHeader->FrameData.U8Data[43] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA43;
			Can_MailboxHeader->FrameData.U8Data[44] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA44;
			Can_MailboxHeader->FrameData.U8Data[45] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA45;
			Can_MailboxHeader->FrameData.U8Data[46] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA46;
			Can_MailboxHeader->FrameData.U8Data[47] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA47;

			Can_MailboxHeader->FrameData.U8Data[48] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA48;
			Can_MailboxHeader->FrameData.U8Data[49] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA49;
			Can_MailboxHeader->FrameData.U8Data[50] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA50;
			Can_MailboxHeader->FrameData.U8Data[51] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA51;
			Can_MailboxHeader->FrameData.U8Data[52] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA52;
			Can_MailboxHeader->FrameData.U8Data[53] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA53;
			Can_MailboxHeader->FrameData.U8Data[54] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA54;
			Can_MailboxHeader->FrameData.U8Data[55] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA55;

			Can_MailboxHeader->FrameData.U8Data[56] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA56;
			Can_MailboxHeader->FrameData.U8Data[57] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA57;
			Can_MailboxHeader->FrameData.U8Data[58] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA58;
			Can_MailboxHeader->FrameData.U8Data[59] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA59;
			Can_MailboxHeader->FrameData.U8Data[60] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA60;
			Can_MailboxHeader->FrameData.U8Data[61] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA61;
			Can_MailboxHeader->FrameData.U8Data[62] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA62;
			Can_MailboxHeader->FrameData.U8Data[63] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.DATA63;
		}
		else
		{
			Can_MailboxHeader->Timestamp = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.TIMESTAMP;
			Can_MailboxHeader->Id = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.ID;
			Can_MailboxHeader->FrameData.Data_Length = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DLC;
			Can_MailboxHeader->BRS = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.BRS;

			Can_MailboxHeader->FrameData.U8Data[0] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA0;
			Can_MailboxHeader->FrameData.U8Data[1] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA1;
			Can_MailboxHeader->FrameData.U8Data[2] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA2;
			Can_MailboxHeader->FrameData.U8Data[3] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA3;
			Can_MailboxHeader->FrameData.U8Data[4] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA4;
			Can_MailboxHeader->FrameData.U8Data[5] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA5;
			Can_MailboxHeader->FrameData.U8Data[6] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA6;
			Can_MailboxHeader->FrameData.U8Data[7] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA7;

			Can_MailboxHeader->FrameData.U8Data[8] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA8;
			Can_MailboxHeader->FrameData.U8Data[9] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA9;
			Can_MailboxHeader->FrameData.U8Data[10] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA10;
			Can_MailboxHeader->FrameData.U8Data[11] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA11;
			Can_MailboxHeader->FrameData.U8Data[12] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA12;
			Can_MailboxHeader->FrameData.U8Data[13] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA13;
			Can_MailboxHeader->FrameData.U8Data[14] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA14;
			Can_MailboxHeader->FrameData.U8Data[15] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA15;

			Can_MailboxHeader->FrameData.U8Data[16] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA16;
			Can_MailboxHeader->FrameData.U8Data[17] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA17;
			Can_MailboxHeader->FrameData.U8Data[18] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA18;
			Can_MailboxHeader->FrameData.U8Data[19] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA19;
			Can_MailboxHeader->FrameData.U8Data[20] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA20;
			Can_MailboxHeader->FrameData.U8Data[21] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA21;
			Can_MailboxHeader->FrameData.U8Data[22] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA22;
			Can_MailboxHeader->FrameData.U8Data[23] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA23;

			Can_MailboxHeader->FrameData.U8Data[24] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA24;
			Can_MailboxHeader->FrameData.U8Data[25] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA25;
			Can_MailboxHeader->FrameData.U8Data[26] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA26;
			Can_MailboxHeader->FrameData.U8Data[27] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA27;
			Can_MailboxHeader->FrameData.U8Data[28] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA28;
			Can_MailboxHeader->FrameData.U8Data[29] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA29;
			Can_MailboxHeader->FrameData.U8Data[30] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA30;
			Can_MailboxHeader->FrameData.U8Data[31] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA31;

			Can_MailboxHeader->FrameData.U8Data[32] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA32;
			Can_MailboxHeader->FrameData.U8Data[33] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA33;
			Can_MailboxHeader->FrameData.U8Data[34] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA34;
			Can_MailboxHeader->FrameData.U8Data[35] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA35;
			Can_MailboxHeader->FrameData.U8Data[36] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA36;
			Can_MailboxHeader->FrameData.U8Data[37] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA37;
			Can_MailboxHeader->FrameData.U8Data[38] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA38;
			Can_MailboxHeader->FrameData.U8Data[39] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA39;

			Can_MailboxHeader->FrameData.U8Data[40] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA40;
			Can_MailboxHeader->FrameData.U8Data[41] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA41;
			Can_MailboxHeader->FrameData.U8Data[42] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA42;
			Can_MailboxHeader->FrameData.U8Data[43] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA43;
			Can_MailboxHeader->FrameData.U8Data[44] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA44;
			Can_MailboxHeader->FrameData.U8Data[45] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA45;
			Can_MailboxHeader->FrameData.U8Data[46] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA46;
			Can_MailboxHeader->FrameData.U8Data[47] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA47;

			Can_MailboxHeader->FrameData.U8Data[48] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA48;
			Can_MailboxHeader->FrameData.U8Data[49] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA49;
			Can_MailboxHeader->FrameData.U8Data[50] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA50;
			Can_MailboxHeader->FrameData.U8Data[51] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA51;
			Can_MailboxHeader->FrameData.U8Data[52] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA52;
			Can_MailboxHeader->FrameData.U8Data[53] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA53;
			Can_MailboxHeader->FrameData.U8Data[54] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA54;
			Can_MailboxHeader->FrameData.U8Data[55] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA55;

			Can_MailboxHeader->FrameData.U8Data[56] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA56;
			Can_MailboxHeader->FrameData.U8Data[57] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA57;
			Can_MailboxHeader->FrameData.U8Data[58] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA58;
			Can_MailboxHeader->FrameData.U8Data[59] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA59;
			Can_MailboxHeader->FrameData.U8Data[60] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA60;
			Can_MailboxHeader->FrameData.U8Data[61] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA61;
			Can_MailboxHeader->FrameData.U8Data[62] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA62;
			Can_MailboxHeader->FrameData.U8Data[63] = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->EFF.DATA63;
		}
	}else
	{

	}
	return ret;
}


/**
  *    дCANFD
  *       MBSIZE<1:0>Ϊ 0x0 ʱὫ RAM ֳ 51  8 ֽڸ䣻
  *       MBSIZE<1:0>Ϊ 0x1 ʱὫ RAM ֳ 36  16 ֽڸ䣻
  *       MBSIZE<1:0>Ϊ 0x2 ʱὫ RAM ֳ 23  32 ֽڸ䣻
  *       MBSIZE<1:0>Ϊ 0x3 ʱὫ RAM ֳ 13  64 ֽڸ䡣
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *      Canfd_Sdu CANFD֡ýṹ
  *   ޡ
  */
Can_ReturnType Can_m_FdWrite(const uint8_t Can_Controller_Index,Canfd_Sdu_Type *Canfd_Sdu)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	/* for canfd test */
	if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		Canfd_Sdu->Sdu_Length = 51;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		Canfd_Sdu->Sdu_Length = 36;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		Canfd_Sdu->Sdu_Length = 23;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		Canfd_Sdu->Sdu_Length = 13;
	}
	else
	{
	}
	Can_m_FdMailBoxErase(Can_Controller_Index);
	for (uint8_t mailbox_number = 0; mailbox_number < Canfd_Sdu->Sdu_Length; mailbox_number++)
	{
		if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
		{
			if (Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length > Data_Length_8)
			{
				Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length = Data_Length_8;
			}
			Can_m_FdCopyDataTo8MailBox(Can_Controller_Index, mailbox_number, &Canfd_Sdu->Can_MailboxHeader[mailbox_number]);
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
		{
			if (Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length > Data_Length_16)
			{
				Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length = Data_Length_16;
			}
			Can_m_FdCopyDataTo16MailBox(Can_Controller_Index, mailbox_number, &Canfd_Sdu->Can_MailboxHeader[mailbox_number]);
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
		{
			if (Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length > Data_Length_32)
			{
				Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length = Data_Length_32;
			}
			Can_m_FdCopyDataTo32MailBox(Can_Controller_Index, mailbox_number, &Canfd_Sdu->Can_MailboxHeader[mailbox_number]);
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
		{
			if (Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length > Data_Length_64)
			{
				Canfd_Sdu->Can_MailboxHeader[mailbox_number].FrameData.Data_Length = Data_Length_64;
			}
			Can_m_FdCopyDataTo64MailBox(Can_Controller_Index, mailbox_number, &Canfd_Sdu->Can_MailboxHeader[mailbox_number]);
		}
		else
		{
		}
	}
}





/**
  *    CANFD
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *      Canfd_Sdu CANFD֡ýṹ
  *    ret CAN_OK
  *           CAN_ERROR_CONFIGURATE
  */
Can_ReturnType Can_m_FdRead(const uint8_t Can_Controller_Index, Canfd_Sdu_Type *Canfd_Sdu)
{
	uint8_t sdu_length = 0x00;
	uint8_t can_controller_index = Can_Controller_Index;
	Can_ReturnType ret;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_8MAILBOX_NUMBER; mailbox_count++)
		{
			if (((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mailbox_count)->SFF.CODE == MAIL_RECEIVE_SUCCESS)
			{
				Can_m_FdCopyDataFrom8MailBox(can_controller_index, mailbox_count, &Canfd_Sdu->Can_MailboxHeader[sdu_length]);
				sdu_length++;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_16MAILBOX_NUMBER; mailbox_count++)
		{
			if (((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mailbox_count)->SFF.CODE == MAIL_RECEIVE_SUCCESS)
			{
				Can_m_FdCopyDataFrom16MailBox(can_controller_index, mailbox_count, &Canfd_Sdu->Can_MailboxHeader[sdu_length]);
				sdu_length++;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_32MAILBOX_NUMBER; mailbox_count++)
		{
			if (((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mailbox_count)->SFF.CODE == MAIL_RECEIVE_SUCCESS)
			{
				Can_m_FdCopyDataFrom32MailBox(can_controller_index, mailbox_count, &Canfd_Sdu->Can_MailboxHeader[sdu_length]);
				sdu_length++;
			}
		}
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		for (uint8_t mailbox_count = 0; mailbox_count < MAX_64MAILBOX_NUMBER; mailbox_count++)
		{
			if (((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mailbox_count)->SFF.CODE == MAIL_RECEIVE_SUCCESS)
			{
				Can_m_FdCopyDataFrom64MailBox(can_controller_index, mailbox_count, &Canfd_Sdu->Can_MailboxHeader[sdu_length]);
				sdu_length++;
			}
		}
	}
	else
	{
	}
	Canfd_Sdu->Sdu_Length = sdu_length;
	if (Canfd_Sdu->Sdu_Length != 0x00)
	{
		ret = CAN_OK;
	}
	else
	{
		ret = CAN_ERROR_CONFIGURATE;
	}
	return ret;
}


/**
  *    дCANFD䡣
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *      MailBox_NumberţȡֵΪMailbox_0~Mailbox_50
  *      Can_MailboxHeaderCANFDͷýṹ
  *    ret CAN_OK
  * 		  CAN_BUSY
  *           CAN_ERROR_CONFIGURATE
  */
Can_ReturnType Can_m_FdMailBox_Write(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	Can_ReturnType ret = CAN_BUSY;
	Can_ReturnType (*func)(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader) = NULL_PTR;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;

	if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		func = &Can_m_FdCopyDataTo8MailBox;
		ret = CAN_OK;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		func = &Can_m_FdCopyDataTo16MailBox;
		ret = CAN_OK;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		func = &Can_m_FdCopyDataTo32MailBox;
		ret = CAN_OK;
	}
	else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		func = &Can_m_FdCopyDataTo64MailBox;
		ret = CAN_OK;
	}
	else
	{
		ret = CAN_ERROR_CONFIGURATE;
	}
	if(ret == CAN_OK)
	{
		ret = func(Can_Controller_Index, MailBox_Number,Can_MailboxHeader);
	}
	return ret;
}


/**
  *    CANFD䡣
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *      MailBox_NumberţȡֵΪMailbox_0~Mailbox_50
  *      Can_MailboxHeaderCANFDͷýṹ
  *    ret CAN_OK    
  * 		  CAN_BUSY  æµ
  *           CAN_ERROR_CONFIGURATE ô
  *           CAN_ERROR_TIMEOUT ߴʱ
  */
Can_ReturnType Can_m_FdMailBox_Read(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number,Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	Can_ReturnType ret = CAN_OK;
	Can_ReturnType (*func)(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number, Canfd_MailboxHeaderType *Can_MailboxHeader) = NULL_PTR;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;

	if ((ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH) && (((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE == MAIL_RECEIVE_SUCCESS))
	{
		func = &Can_m_FdCopyDataFrom8MailBox;
		ret = CAN_OK;
	}
	else if ((ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH) && (((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE == MAIL_RECEIVE_SUCCESS))
	{
		func = &Can_m_FdCopyDataFrom16MailBox;
		ret = CAN_OK;
	}
	else if ((ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH) && (((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE == MAIL_RECEIVE_SUCCESS))
	{
		func = &Can_m_FdCopyDataFrom32MailBox;
		ret = CAN_OK;
	}
	else if ((ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH) && (((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE == MAIL_RECEIVE_SUCCESS))
	{
		func = &Can_m_FdCopyDataFrom64MailBox;
		ret = CAN_OK;
	}else
	{
		ret = CAN_ERROR_CONFIGURATE;
	}

	if(ret == CAN_OK)
	{
		ret = func(Can_Controller_Index, MailBox_Number,Can_MailboxHeader);
	}else
	{
	}
	return ret;
}


/**
  *    CANFDֱͣз䷢ɡ
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *    ret CAN_OK    
  * 		  CAN_BUSY  æµ
  *           CAN_ERROR_CONFIGURATE ô
  *           CAN_ERROR_TIMEOUT ߴʱ
  */
Can_ReturnType Can_m_FdTransmitpack(const uint8_t Can_Controller_Index)
{
	Can_ReturnType ret = CAN_ERROR_CONFIGURATE;
	volatile uint32_t delay_count = TRANSMIT_TIMEOUT_COUNT*10;
	uint8_t Mail_Box_Number = 0x00;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	if(ControllerRegPtr->CANFD_CTLR0.B.TXSTA == 0x00)
	{
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		/* waiting for arbitration  end */
		while ((ControllerRegPtr->CANFD_AMSTA.B.ARBSTA0 == 0x00) && (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x00))
			;
		if (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x01)
		{
			/* arbitration  end  and not found  the mailbox */
			ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x01;
			ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x00;

		}

		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x01;
		if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
		{
			Mail_Box_Number = 51;
			for (uint8_t mail_box_count = 0; mail_box_count < Mail_Box_Number; mail_box_count++)
			{
				while(1)
				{
					volatile DataBuffer_8byteType *MailBox = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mail_box_count);
					if(MailBox->SFF.CODE != MAIL_TRANSMIT || (!(delay_count--)))
					{
						if(MailBox->SFF.CODE == MAIL_TRANSMIT)
						{
							ret = CAN_ERROR_TIMEOUT;
							return ret;
						}else
						{
							break;
						}
					}
				}
			}
			ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
			ret = CAN_OK;
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
		{
			Mail_Box_Number = 36;
			for (uint8_t mail_box_count = 0; mail_box_count < Mail_Box_Number; mail_box_count++)
			{
				while(1)
				{
					volatile DataBuffer_16byteType *MailBox = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mail_box_count);
					if(MailBox->SFF.CODE != MAIL_TRANSMIT || (!(delay_count--)))
					{
						if(MailBox->SFF.CODE == MAIL_TRANSMIT)
						{
							ret = CAN_ERROR_TIMEOUT;
							return ret;
						}else
						{
							break;
						}
					}
				}
			}
			ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
			ret = CAN_OK;
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
		{
			Mail_Box_Number = 23;
			for (uint8_t mail_box_count = 0; mail_box_count < Mail_Box_Number; mail_box_count++)
			{
				while(1)
				{
					volatile DataBuffer_32byteType *MailBox = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mail_box_count);
					if(MailBox->SFF.CODE != MAIL_TRANSMIT || (!(delay_count--)))
					{
						if(MailBox->SFF.CODE == MAIL_TRANSMIT)
						{
							ret = CAN_ERROR_TIMEOUT;
							return ret;
						}else
						{
							break;
						}
					}
				}
			}
			ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
			ret = CAN_OK;
		}
		else if (ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
		{
			Mail_Box_Number = 13;
			for (uint8_t mail_box_count = 0; mail_box_count < Mail_Box_Number; mail_box_count++)
			{
				while(1)
				{
					volatile DataBuffer_64byteType *MailBox = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + mail_box_count);
					if(MailBox->SFF.CODE != MAIL_TRANSMIT || (!(delay_count--)))
					{
						if(MailBox->SFF.CODE == MAIL_TRANSMIT)
						{
							ret = CAN_ERROR_TIMEOUT;
							return ret;
						}else
						{
							break;
						}
					}
				}
			}
			ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
			ret = CAN_OK;
		}
		else
		{
			ret = CAN_ERROR_CONFIGURATE;
		}
	}else
	{
		ret = CAN_BUSY;
	}
	return ret;
}


/**
  *    CANFD䵥ηֱͣٲóɹ䷢ɡ
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *    ret CAN_OK    
  * 		  CAN_BUSY  æµ
  *           CAN_ERROR_CONFIGURATE ô
  *           CAN_ERROR_TIMEOUT ߴʱ
  */
Can_ReturnType Can_m_FdTransmitonce(const uint8_t Can_Controller_Index)
{
	Can_ReturnType ret = CAN_ERROR_CONFIGURATE;
	volatile uint32_t delay_count = TRANSMIT_TIMEOUT_COUNT;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	/* waiting for arbitration  end */
	while ((ControllerRegPtr->CANFD_AMSTA.B.ARBSTA0 == 0x00) && (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x00))
		;
	if (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x01)
	{
		/* arbitration  end  and not found  the mailbox */
		ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x01;
		ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x00;
	}
	if(ControllerRegPtr->CANFD_IFR.B.MOENDIF == 0x01)
	{
		arbwinnermailboxnumber = (uint8_t )(ControllerRegPtr->CANFD_AMSTA.B.ARBWINNER >>4);
		/* arbitration end and found arbitration success mailbox */
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x01;
		while ((ControllerRegPtr->CANFD_CTLR0.B.TXSTA == 0x00) && (delay_count--))
			;
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		if(ControllerRegPtr->CANFD_CTLR0.B.TXSTA == 0x01)
		{
			while(((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + arbwinnermailboxnumber)->SFF.CODE == MAIL_TRANSMIT)
				;
			if(ControllerRegPtr->CANFD_DATA.DataBuffer_8byte.SFF.ID != ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + arbwinnermailboxnumber)->SFF.ID )
			{
				a156canfd_txflag = 0x01;
			}
			ret = CAN_OK;
		}else
		{
			ret = CAN_ERROR_TIMEOUT;
		}

	}else
	{
		ret = CAN_BUSY;
	}
	return ret;
}

/**
  *    CANFDĿ䣬ֱĿ귢䷢ɡ
  *    Can_Controller_Index: CANFDţȡֵΪ CANfd6CANfd7
  *    ret CAN_OK    
  * 		  CAN_BUSY  æµ
  *           CAN_ERROR_CONFIGURATE ô
  *           CAN_ERROR_TIMEOUT ߴʱ
  */
/*
 * wait until target transmit mailbox tansmited
 * */
Can_ReturnType Can_m_FdTransmituntil(const uint8_t Can_Controller_Index,const uint8_t Mailbox_Number)
{
	Can_ReturnType ret = CAN_ERROR_CONFIGURATE;
	volatile uint32_t delay_count = TRANSMIT_TIMEOUT_COUNT;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;

	/* waiting for arbitration  end */
	while ((ControllerRegPtr->CANFD_AMSTA.B.ARBSTA0 == 0x00) && (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x00))
		;
	if (ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1 == 0x01)
	{
		/* arbitration  end  and not found  the mailbox */
		ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x01;
		ControllerRegPtr->CANFD_CTLR1.B.ARBSTART = 0x00;
	}
	/* arbitration end and found arbitration success mailbox */
	ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x01;
	if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		volatile DataBuffer_8byteType *MailBox = ((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + Mailbox_Number);
		while (MailBox->SFF.CODE != MAIL_TRANSMIT_SUCCESS);
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		if(MailBox->SFF.CODE == MAIL_TRANSMIT_SUCCESS)
		{
			ret = CAN_OK;
		}else if(delay_count == 0xFFFF)
		{
			ret = CAN_ERROR_TIMEOUT;
		}else
		{
			ret = CAN_ERROR_CONFIGURATE;
		}
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		volatile DataBuffer_16byteType *MailBox = ((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + Mailbox_Number);
		while (MailBox->SFF.CODE != MAIL_TRANSMIT_SUCCESS);
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		if(MailBox->SFF.CODE == MAIL_TRANSMIT_SUCCESS)
		{
			ret = CAN_OK;
		}else if(delay_count == 0xFFFF)
		{
			ret = CAN_ERROR_TIMEOUT;
		}else
		{
			ret = CAN_ERROR_CONFIGURATE;
		}
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		volatile DataBuffer_32byteType *MailBox = ((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + Mailbox_Number);
		while (MailBox->SFF.CODE != MAIL_TRANSMIT_SUCCESS);
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		if(MailBox->SFF.CODE == MAIL_TRANSMIT_SUCCESS)
		{
			ret = CAN_OK;
		}else if(delay_count == 0xFFFF)
		{
			ret = CAN_ERROR_TIMEOUT;
		}else
		{
			ret = CAN_ERROR_CONFIGURATE;
		}
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		volatile DataBuffer_64byteType *MailBox = ((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + Mailbox_Number);
		while (MailBox->SFF.CODE != MAIL_TRANSMIT_SUCCESS);
		ControllerRegPtr->CANFD_CTLR0.B.TXR = 0x00;
		if(MailBox->SFF.CODE == MAIL_TRANSMIT_SUCCESS)
		{
			ret = CAN_OK;
		}else if(delay_count == 0xFFFF)
		{
			ret = CAN_ERROR_TIMEOUT;
		}else
		{
			ret = CAN_ERROR_CONFIGURATE;
		}
	}
	return ret;
}


/**
  *   CANFDȡжϱ־λ
  *    Can_Controller_Index : CANFDţȡֵΪCANfd6CANfd7
  *      Can_Controller_InterruptFlag CANFDжϱ־λ
  *      			RxIntFlagжϱ־λ
  *      			TxIntFlagжϱ־λ
  *      			BusOffFlag߹رжϱ־λ
  *      			WakeUpFlagжϱ־λ
  *      			ErrorAlarmFlag󱨾жϱ־λ
  *      			OverFlowFlagжϱ־λ
  *					ErrorNegativeFlagжϱ־λ
  *					ArbitrateLoseFlagռʧжϱ־λ
  *					BusErrorFlagߴжϱ־λ
  *					DmaTXDFlagDMAжϱ־λ
  *					DmaRXDFlagDMAжϱ־λ
  *					RXBStateFlag״̬־λ
  *					TrigerMBFlagմжϱ־λ
  *					TransmitDelayOffsetFailFlagʱʧܱ־λ
  *					ArbitrateFailFlagٲʧܱ־λ
  *					MoveOutEndFlagmove out жϱ־λ
  *		 rmc_countֵ
  *   ޡ
  */
void Can_m_FdGetIntFlag(const uint8_t Can_Controller_Index,Can_Controller_InterruptFlagType *Can_Controller_InterruptFlag, uint8_t *rmc_count)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_Controller_InterruptFlag->ArbitrateFailFlag = ControllerRegPtr->CANFD_AMSTA.B.ARBSTA1;
	Can_Controller_InterruptFlag->ArbitrateLoseFlag = ControllerRegPtr->CANFD_IFR.B.ALIF;
	Can_Controller_InterruptFlag->BusErrorFlag = ControllerRegPtr->CANFD_IFR.B.BEIF;
	Can_Controller_InterruptFlag->BusOffFlag = ControllerRegPtr->CANFD_IFR.B.BOFFIF;
	Can_Controller_InterruptFlag->DmaRXDFlag = ControllerRegPtr->CANFD_IFR.B.CRXDF;
	Can_Controller_InterruptFlag->DmaTXDFlag = ControllerRegPtr->CANFD_IFR.B.CTXDF;
	Can_Controller_InterruptFlag->ErrorAlarmFlag = ControllerRegPtr->CANFD_IFR.B.EAIF;
	Can_Controller_InterruptFlag->ErrorNegativeFlag = ControllerRegPtr->CANFD_IFR.B.ENIF;
	Can_Controller_InterruptFlag->MoveOutEndFlag = ControllerRegPtr->CANFD_IFR.B.MOENDIF;
	Can_Controller_InterruptFlag->OverFlowFlag = ControllerRegPtr->CANFD_IFR.B.DOVFIF;
	Can_Controller_InterruptFlag->RXBStateFlag = ControllerRegPtr->CANFD_IFR.B.RXBSTAIF;
	Can_Controller_InterruptFlag->RxIntFlag = ControllerRegPtr->CANFD_IFR.B.CANRXIF;
	Can_Controller_InterruptFlag->TransmitDelayOffsetFailFlag = ControllerRegPtr->CANFD_IFR.B.TDCFAILIF;
	Can_Controller_InterruptFlag->TrigerMBFlag = ControllerRegPtr->CANFD_IFR.B.TRGMBIF;
	Can_Controller_InterruptFlag->TxIntFlag = ControllerRegPtr->CANFD_IFR.B.CANTXIF;
	Can_Controller_InterruptFlag->WakeUpFlag = ControllerRegPtr->CANFD_IFR.B.WUIF;

	/* get canfd controller receive message number */

	*rmc_count = ControllerRegPtr->CANFD_CTLR0.B.CANRMC;
}


/**
  *   CANFDжϱ־λ
  *    Can_Controller_Index : CANFDţȡֵΪCANfd6CANfd7
  *      Can_Controller_InterruptFlag CANFDжϱ־λ
  *      			RxIntFlagжϱ־λ
  *      			TxIntFlagжϱ־λ
  *      			BusOffFlag߹رжϱ־
  *      			WakeUpFlagжϱ־λ
  *      			ErrorAlarmFlag󱨾жϱ־λ
  *      			OverFlowFlagжϱ־λ
  *					ErrorNegativeFlagжϱ־λ
  *					ArbitrateLoseFlagռʧжϱ־λ
  *					BusErrorFlagߴжϱ־λ
  *					DmaTXDFlagDMAжϱ־λ
  *					DmaRXDFlagDMAжϱ־λ
  *					RXBStateFlag״̬־λ
  *					TrigerMBFlagմжϱ־λ
  *					TransmitDelayOffsetFailFlagʱʧܱ־λ
  *					ArbitrateFailFlagٲʧܱ־λ
  *					MoveOutEndFlagmove out жϱ־λ
  *		 rmc_countֵ
  *   ޡ
  */
void Can_m_FdClearIntFlag(const uint8_t Can_Controller_Index, Can_Controller_InterruptFlagType *Can_Controller_InterruptFlag,uint8_t rmc_count)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	uint32_t Rcr_Value = 0;
	if (Can_Controller_InterruptFlag->ArbitrateLoseFlag == 0x01)
	{
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		ControllerRegPtr->CANFD_IER.B.ALIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.ALIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.ALIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->BusErrorFlag == 0x01)
	{
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		ControllerRegPtr->CANFD_IER.B.BEIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.BEIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.BEIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->BusOffFlag == 0x01)
	{
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		ControllerRegPtr->CANFD_IER.B.BOFFIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.BOFFIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.BOFFIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->ErrorAlarmFlag == 0x01)
	{
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		ControllerRegPtr->CANFD_IER.B.EAIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.EAIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.EAIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->ErrorNegativeFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.ENIC = 0x01;
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		while (ControllerRegPtr->CANFD_IFR.B.ENIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.ENIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->OverFlowFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.DOVFIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.DOVFIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.DOVFIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->TransmitDelayOffsetFailFlag == 0x01)
	{
		Rcr_Value = ControllerRegPtr->CANFD_RCR.R;
		ControllerRegPtr->CANFD_IER.B.TDCFAILIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.TDCFAILIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.TDCFAILIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->TrigerMBFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.TRGMBIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.TRGMBIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.TRGMBIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->TxIntFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.CANTXIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.CANTXIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.CANTXIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->RxIntFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.CANRXIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.CANRXIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.CANRXIC = 0x00;
	}

	if (Can_Controller_InterruptFlag->WakeUpFlag == 0x01)
	{
		ControllerRegPtr->CANFD_IER.B.WUIC = 0x01;
		while (ControllerRegPtr->CANFD_IFR.B.WUIF == 0x01)
			;
		ControllerRegPtr->CANFD_IER.B.WUIC = 0x00;
	}

	for(uint8_t release_count = 0;release_count<rmc_count;release_count++)
	{
		ControllerRegPtr->CANFD_CTLR0.B.RELRX = 0x01;
	}
}




/**
  *    CANFDá
  *    Can_Controller_Index : CANFDţȡֵΪCANfd6CANfd7
  *      MailBox_NumberţȡֵΪMailbox_0~Mailbox_50
  *      Can_MailboxHeaderͷýṹ
  *    ret CAN_OK
  * 		  CAN_BUSY
  */
Can_ReturnType Can_m_FdMailBoxCode_Set(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number,Canfd_MailboxHeaderType *Can_MailboxHeader)
{
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Code_Segment Ret_Code_Segment = {0};
	Can_ReturnType ret = CAN_BUSY;
	Ret_Code_Segment = Can_m_FdCsCodeSet(Can_MailboxHeader);
	ret = Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ret == CAN_OK)
	{
		if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
		{
			((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
		}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
		{
			((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
		}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
		{
			((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
		}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
		{
			((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CS_CODE = Ret_Code_Segment.CS_CODE;
		}
	}else
	{

	}
	return ret;
}




/**
  *    ȡCANFD״̬
  *    Can_Controller_Index : CANFDţȡֵΪCANfd6CANfd7
  *      MailBox_NumberţȡֵΪMailbox_0~Mailbox_50
  *    Ret_State MAIL_UNDEFINED δ
  *     			MAIL_RECEIVE 
  *     			MAIL_RECEIVE_SUCCESS ճɹ
  *     			MAIL_TRANSMIT ䷢
  *     			MAIL_TRANSMIT_SUCCESS ䷢ͳɹ
  *     			MAIL_RTR_REQUEST Զ֡
  *     			MAIL_RTR_ANSWER Զ֡ظ
  */
Can_MailBoxType Can_m_FdGetMailBoxState(const uint8_t Can_Controller_Index, const Can_MailboxNumType MailBox_Number)
{
	Can_MailBoxType Ret_State = MAIL_UNDEFINED;
	volatile Kf32a_Canfd_Reg *ControllerRegPtr = (Kf32a_Canfd_Reg *)Can_m_ControllersInfo[Can_Controller_Index].BaseAddress;
	Can_m_FdWaitConfilct(Can_Controller_Index);
	if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_8_BYTE_DATALENGTH)
	{
		Ret_State = (Can_MailBoxType)(((DataBuffer_8byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE);
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_16_BYTE_DATALENGTH)
	{
		Ret_State = (Can_MailBoxType)(((DataBuffer_16byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE);
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_32_BYTE_DATALENGTH)
	{
		Ret_State = (Can_MailBoxType)(((DataBuffer_32byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE);
	}else if(ControllerRegPtr->CANFD_CTLR0.B.MBSIZE == CAN_64_BYTE_DATALENGTH)
	{
		Ret_State = (Can_MailBoxType)(((DataBuffer_64byteType *)(Can_m_ControllersInfo[Can_Controller_Index].MBBaseAddress) + MailBox_Number)->SFF.CODE);
	}
	return Ret_State;
}




/**
  * @brief This function handles transmit busy or transmit error
  */
void Can_m_FdTransmiterrorHandler(void)
{
/*
 *
 * */
}

