/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : KF32A156_gtim.c                                       *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This file provides general timer peripherals 		  *
 *  					function, including:								  *
 *          			+ Generic Timer (GPTIM) initialization and 			  *
 *          			  configuration function definition					  *
 *          			+ Generic Timer (GPTIM) interrupt and flag management *
 *          			  function definition    							  *
 *          			+ Definition of general capture/comparison/PWM 		  *
 *          			  peripheral (CCP) capture function 				  *
 *          			+ Definition of general catch/compare/PWM peripherals *
 *          			  (CCP) interrupt and flag management functions  	  *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/

/******************************************************************************
**                      		Include Files                                **
******************************************************************************/

#include "kf32a156_gtim.h"
#include "kf32a156_rst.h"
#include "kf32a156_pclk.h"

/*******************************************************************************
**                         (GTIM) Module private definition  	              **
*******************************************************************************/

/* GPTIMx_CTL1Ĵ */
#define GPTIM_CTL1_INIT_MASK            (GPTIM_CTL1_TXCLK \
                                       | GPTIM_CTL1_TXCMS \
                                       | GPTIM_CTL1_TXSY \
                                       | GPTIM_CTL1_TXCS)

/* GPTIMx_CTL2Ĵ */
#define GPTIM_CTL2_INIT_MASK            (GPTIM_CTL2_TXMMS \
                                       | GPTIM_CTL2_TXSMS \
                                       | GPTIM_CTL2_TXMSSYNC)

/* CCPXCTL3Ĵ׽ģʽ */
#define CCP_CTL3_CAP_INIT_MASK          (CCP_CTL3_PXPWMI | CCP_CTL3_PXXORM)

/* CCPXCTL3ĴPWM */
#define CCP_CTL3_PWM_INIT_MASK          (CCP_CTL3_PXSPM | CCP_CTL3_PXSPMST)

/* CCPXCYĴƫƵַ */
#define CCP_CY_OFFSET                   ((uint32_t)0x20)

/* CCPXRYĴƫƵַ */
#define CCP_RY_OFFSET                   ((uint32_t)0x44)



/**
  *   ##### ʱ踴λ #####
  */

/**
  *   ʱ踴λʹʱӡ
  *   TIMx: ʱڴṹָ룬ȡֵΪ T0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/
  *           T5_SFR/T6_SFR/T7_SFR/T8_SFR/T18_SFR/T19_SFR/T21_SFR
  *   
  */
void GPTIM_Reset (GPTIM_SFRmap*  GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T0_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T0RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T0RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_T0CLKEN, TRUE);
    }
    else if (GPTIMx == T1_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T1RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T1RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_T1CLKEN, TRUE);
    }
    else if (GPTIMx == T2_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T2RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T2RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_T2CLKEN, TRUE);
    }
    else if (GPTIMx == T3_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T3RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T3RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_T3CLKEN, TRUE);
    }
    else if (GPTIMx == T4_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T4RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_T4RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_T4CLKEN, TRUE);
    }
    else if (GPTIMx == T18_SFR)
    {
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T18RST, TRUE);
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T18RST, FALSE);
        PCLK_CTL2_Peripheral_Clock_Enable(PCLK_CTL2_T18CLKEN, TRUE);
    }
    else if (GPTIMx == T19_SFR)
    {
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T19RST, TRUE);
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T19RST, FALSE);
        PCLK_CTL2_Peripheral_Clock_Enable(PCLK_CTL2_T19CLKEN, TRUE);
    }
    else if (GPTIMx == T21_SFR)
    {
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T21RST, TRUE);
        RST_CTL2_Peripheral_Reset_Enable(RST_CTL2_T21RST, FALSE);
        PCLK_CTL2_Peripheral_Clock_Enable(PCLK_CTL2_T21CLKEN, TRUE);
    }
    else
    {
        ;
    }
}

/**
  *   ##### ʱ踴λ #####
  */

/**
  *   ##### ͨöʱ(GPTIM)ʼú #####
  */
/**
  *   ͨöʱ(GPTIM)á
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       gptimInitStruct: ͨöʱϢṹָ롣
  *   
  */
void GPTIM_Configuration(GPTIM_SFRmap* GPTIMx,
                    GPTIM_InitTypeDef* gptimInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_COUNTER_MODE(gptimInitStruct->m_CounterMode));
    CHECK_RESTRICTION(CHECK_GPTIM_CLOCK_CONFIG(gptimInitStruct->m_Clock));
    CHECK_RESTRICTION(CHECK_GPTIM_WORK_MODE(gptimInitStruct->m_WorkMode));
    CHECK_RESTRICTION(CHECK_GPTIM_MASTER_MODE(gptimInitStruct->m_MasterMode));
    CHECK_RESTRICTION(CHECK_GPTIM_SLAVE_MODE(gptimInitStruct->m_SlaveMode));
    CHECK_RESTRICTION(CHECK_GPTIM_SYNC_MODE(gptimInitStruct->m_EXPulseSync));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(gptimInitStruct->m_MasterSlaveSync));

    if (GPTIMx == T21_SFR)
    {
        /*---------------- TxCNTĴ ----------------*/
        GPTIMx->CNT = gptimInitStruct->m_Counter;

        /*--------------- PPxڼĴ ---------------*/
        GPTIMx->PPX = gptimInitStruct->m_Period;

        /*-------------- TxԤƵĴ ---------------*/
        GPTIMx->PRSC = gptimInitStruct->m_Prescaler;
    }
    else
    {
        /*---------------- TxCNTĴ ----------------*/
        GPTIMx->CNT = (uint16_t)gptimInitStruct->m_Counter;

        /*--------------- PPxڼĴ ---------------*/
        GPTIMx->PPX = (uint16_t)gptimInitStruct->m_Period;

        /*-------------- TxԤƵĴ ---------------*/
        GPTIMx->PRSC = (uint16_t)gptimInitStruct->m_Prescaler;
    }

    /*------------------ TXCTL1Ĵ -----------------*/
    /* ݽṹԱm_CounterModeTXCMSλ */
    /* ݽṹԱm_ClockTXCLKλ */
    /* ݽṹԱm_WorkModeTXCSλ */
    /* ݽṹԱm_EXPulseSyncTXSYλ */
    tmpreg = ((uint32_t)gptimInitStruct->m_CounterMode) | \
             ((uint32_t)gptimInitStruct->m_Clock) | \
             ((uint32_t)gptimInitStruct->m_WorkMode) | \
             ((uint32_t)gptimInitStruct->m_EXPulseSync);
    GPTIMx->CTL1 = SFR_Config (GPTIMx->CTL1,
                          ~GPTIM_CTL1_INIT_MASK,
                          tmpreg);

    /*-------------------- TXCTL2Ĵ --------------------*/
    /* ݽṹԱm_MasterModeTXMMSλ */
    /* ݽṹԱm_SlaveModeTXSMSλ */
    /* ݽṹԱm_MasterSlaveSyncTXMSSYNCλ */
    tmpreg = ((uint32_t)gptimInitStruct->m_MasterMode) | \
             ((uint32_t)gptimInitStruct->m_SlaveMode) | \
             ((uint32_t)gptimInitStruct->m_MasterSlaveSync << GPTIM_CTL2_TXMSSYNC_POS);
    GPTIMx->CTL2 = SFR_Config (GPTIMx->CTL2,
                          ~GPTIM_CTL2_INIT_MASK,
                          tmpreg);
}

/**
  *   ʼͨöʱϢṹ塣
  *   gptimInitStruct: ָʼĽṹָ롣
  *   
  */
void
GPTIM_Struct_Init (GPTIM_InitTypeDef* gptimInitStruct)
{
    /* Txֵ */
    gptimInitStruct->m_Counter = 0x00000000;
    /* Txֵ */
    gptimInitStruct->m_Period = 0xFFFFFFFF;
    /* TxԤƵֵ */
    gptimInitStruct->m_Prescaler = 0x00000000;
    /* Txģʽ */
    gptimInitStruct->m_CounterMode = GPTIM_COUNT_UP_OF;
    /* Txʱ */
    gptimInitStruct->m_Clock = GPTIM_SCLK;
    /* öʱ/ģʽѡ */
    gptimInitStruct->m_WorkMode = GPTIM_TIMER_MODE;
    /* ģʽѡ */
    gptimInitStruct->m_MasterMode = GPTIM_MASTER_TXUR_SIGNAL;
    /* ôģʽѡ */
    gptimInitStruct->m_SlaveMode = GPTIM_SLAVE_FORBIDDEN_MODE;
    /* Txģʽⲿͬ */
    gptimInitStruct->m_EXPulseSync = GPTIM_EX_SYNC_MODE;
    /* ģʽͬ */
    gptimInitStruct->m_MasterSlaveSync = FALSE;
}

/**
  *   ʱʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: ʱʹܿƣ
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Cmd (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL1ĴTXENλ --------------------*/
    if (NewState != FALSE)
    {

        SFR_SET_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXEN_POS);
    }
    else
    {

        SFR_CLR_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXEN_POS);
    }
}

/**
  *   ¶ʱֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       Counter: µļֵȡֵ16λݡ
  *   
  */
void
GPTIM_Set_Counter (GPTIM_SFRmap* GPTIMx, uint32_t Counter)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /*---------------- TxCNTĴ ----------------*/
        GPTIMx->CNT = Counter;
    }
    else
    {
        /*---------------- TxCNTĴ ----------------*/
        GPTIMx->CNT = (uint16_t)Counter;
    }
}

/**
  *   ¶ʱֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       Period: µֵȡֵ16λݡ
  *   
  */
void
GPTIM_Set_Period (GPTIM_SFRmap* GPTIMx, uint32_t Period)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /*---------------- TxPPXĴ ----------------*/
        GPTIMx->PPX = Period;
    }
    else
    {
        /*---------------- TxPPXĴ ----------------*/
        GPTIMx->PPX = (uint16_t)Period;
    }
}

/**
  *   ¶ʱԤƵֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       Prescaler: µԤƵȡֵ32λ16λݡ
  *   
  */
void
GPTIM_Set_Prescaler (GPTIM_SFRmap* GPTIMx, uint32_t Prescaler)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /*---------------- TxPRSCĴ ----------------*/
        GPTIMx->PRSC = Prescaler;
    }
    else
    {
        /*---------------- TxPRSCĴ ----------------*/
        GPTIMx->PRSC = (uint16_t)Prescaler;
    }
}

/**
  *   ¶ʱģʽ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       CounterMode: µļģʽ
  *                    ȡֵΧΪ
  *                      GPTIM_COUNT_DOWN_UF: ¼,жϱ־
  *                      GPTIM_COUNT_UP_OF: ϼ,жϱ־
  *                      GPTIM_COUNT_UP_DOWN_OF: -¼,жϱ־
  *                      GPTIM_COUNT_UP_DOWN_UF: -¼,жϱ־
  *                      GPTIM_COUNT_UP_DOWN_OUF: -¼,жϱ־
  *   
  */
void
GPTIM_Counter_Mode_Config (GPTIM_SFRmap* GPTIMx, uint32_t CounterMode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_COUNTER_MODE(CounterMode));

    /*-------------------- TXCTL1ĴTXCMSλ --------------------*/
    GPTIMx->CTL1 = SFR_Config (GPTIMx->CTL1,
                          ~GPTIM_CTL1_TXCMS,
                          CounterMode);
}

/**
  *   ¶ʱʱӡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewClock: µĹʱӣ
  *                 ȡֵΧΪ
  *                   GPTIM_SCLK: ѡSCLKʱ
  *                   GPTIM_HFCLK: ѡHFCLKʱ
  *                   GPTIM_LFCLK: ѡLFCLKʱ
  *                   GPTIM_T0_INTLF: ͹ĶʱT0ѡINTLFʱ
  *   
  */
void
GPTIM_Clock_Config (GPTIM_SFRmap* GPTIMx, uint32_t NewClock)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_CLOCK_CONFIG(NewClock));

    /*-------------------- TXCTL1ĴTXCLKλ --------------------*/
    GPTIMx->CTL1 = SFR_Config (GPTIMx->CTL1,
                          ~GPTIM_CTL1_TXCLK,
                          NewClock);
}

/**
  *   ¶ʱģʽⲿͬλ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewClock: µĶʱģʽⲿͬλʹ״̬
  *                 ȡֵΧΪ
  *                   GPTIM_EX_SYNC_MODE: ⲿͬ
  *                   GPTIM_NO_SYNC_MODE: ⲿͬ
  *   
  */
void
GPTIM_External_Pulse_Sync_Config (GPTIM_SFRmap* GPTIMx, uint32_t PulseSync)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_SYNC_MODE(PulseSync));

    /*-------------------- TXCTL1ĴTXSYλ --------------------*/
    if (PulseSync != GPTIM_EX_SYNC_MODE)
    {
        /* ⲿͬ */
        SFR_SET_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXSY_POS);
    }
    else
    {
        /* ⲿͬ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXSY_POS);
    }
}

/**
  *   ¶ʱ/ģʽѡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: µĶʱ/ģʽ
  *                 ȡֵΧΪ
  *                   GPTIM_TIMER_MODE: ʱģʽ
  *                   GPTIM_COUNTER_MODE: ģʽ
  *   
  */
void
GPTIM_Work_Mode_Config (GPTIM_SFRmap* GPTIMx, uint32_t NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_WORK_MODE(NewState));

    /*-------------------- TXCTL1ĴTXCSλ --------------------*/
    if (NewState != GPTIM_TIMER_MODE)
    {
        /* ģʽ */
        SFR_SET_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXCS_POS);
    }
    else
    {
        /* ʱģʽ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXCS_POS);
    }
}

/**
  *   ¿λ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: ʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Updata_Immediately_Config (GPTIM_SFRmap* GPTIMx,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴTXURλ --------------------*/
    if (NewState != FALSE)
    {
        /* ʹ */
        SFR_SET_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUR_POS);
    }
    else
    {
        /* δʹ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUR_POS);
    }
}

/**
  *   ģʽͬλ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: ģʽͬλ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Master_Slave_Snyc_Config (GPTIM_SFRmap* GPTIMx,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴTXMSSYNCλ --------------------*/
    if (NewState != FALSE)
    {
        /* ʹ */
        SFR_SET_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXMSSYNC_POS);
    }
    else
    {
        /* δʹ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXMSSYNC_POS);
    }
}

/**
  *   ôѡλ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       TriggerSelect: ѡ
  *                      ȡֵΧΪ
  *                         GPTIM_TRIGGER_T1
  *                         GPTIM_TRIGGER_T2
  *                         GPTIM_TRIGGER_T3
  *                         GPTIM_TRIGGER_T4
  *                         GPTIM_TRIGGER_T5
  *                         GPTIM_TRIGGER_T9
  *                         GPTIM_TRIGGER_T14
  *                         GPTIM_TRIGGER_T15
  *                         GPTIM_TRIGGER_T18
  *                         GPTIM_TRIGGER_T19
  *                         GPTIM_TRIGGER_T20
  *                         GPTIM_TRIGGER_T21
  *                         GPTIM_TRIGGER_TXCK
  *                         GPTIM_TRIGGER_CCPXCH1
  *                         GPTIM_TRIGGER_CCPXCH2
  *                         GPTIM_TRIGGER_CCPXCH3
  *   
  */
void
GPTIM_Trigger_Select_Config (GPTIM_SFRmap* GPTIMx,
                    uint32_t TriggerSelect)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_TRIGGER_MODE(TriggerSelect));

    /*-------------------- TXCTL2ĴTXTSλ --------------------*/
    GPTIMx->CTL2 = SFR_Config (GPTIMx->CTL2,
                          ~GPTIM_CTL2_TXTS,
                          TriggerSelect);
}

/**
  *   ôģʽѡλ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       SlaveSelect: ģʽѡ
  *                    ȡֵΧΪ
  *                      GPTIM_SLAVE_FORBIDDEN_MODE: ģʽֹ
  *                      GPTIM_SLAVE_TRIGGER_MODE: ģʽ
  *                      GPTIM_SLAVE_GATED_MODE: ſģʽ
  *                      GPTIM_SLAVE_RESET_MODE: λģʽ
  *                      GPTIM_SLAVE_COUNTER_MODE: ģʽ2
  *   
  */
void
GPTIM_Slave_Mode_Config (GPTIM_SFRmap* GPTIMx, uint32_t SlaveMode)
{
   /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_SLAVE_MODE(SlaveMode));

    /*-------------------- TXCTL2ĴTXSMSλ --------------------*/
    GPTIMx->CTL2 = SFR_Config (GPTIMx->CTL2,
                          ~GPTIM_CTL2_TXSMS,
                          SlaveMode);
}

/**
  *   ģʽѡλ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       MasterMode: ģʽѡ
  *                   ȡֵΧΪ
  *                     GPTIM_MASTER_TXUR_SIGNAL: URλΪ
  *                     GPTIM_MASTER_TXEN_SIGNAL: TXENΪ
  *                     GPTIM_MASTER_TXIF_SIGNAL: TXIFΪ
  *                     GPTIM_MASTER_CCPXCH1IF_SIGNAL: CCPxCH1IFΪ
  *                     GPTIM_MASTER_CCPXCH1_SIGNAL: CCPxCH1 Ϊ
  *                     GPTIM_MASTER_CCPXCH2_SIGNAL: CCPxCH2 Ϊ
  *                     GPTIM_MASTER_CCPXCH3_SIGNAL: CCPxCH3 Ϊ
  *                     GPTIM_MASTER_CCPXCH4_SIGNAL: CCPxCH4 Ϊ
  *   
  */
void
GPTIM_Master_Mode_Config (GPTIM_SFRmap* GPTIMx, uint32_t MasterMode)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_GPTIM_MASTER_MODE(MasterMode));

    /*-------------------- TXCTL2ĴTXMMSλ --------------------*/
    GPTIMx->CTL2 = SFR_Config (GPTIMx->CTL2,
                          ~GPTIM_CTL2_TXMMS,
                          MasterMode);
}

/**
  *   ظ¼λ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: ¼״̬
  *                 ȡֵΧΪ
  *                   TRUE: 
  *                   FALSE: ÿڸ
  *   
  */
void
GPTIM_Updata_Rising_Edge_Config (GPTIM_SFRmap* GPTIMx,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴTXUDEVTλ --------------------*/
    if (NewState != FALSE)
    {
        /*  */
        SFR_SET_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUDEVT_POS);
    }
    else
    {
        /* ÿڸ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUDEVT_POS);
    }
}

/**
  *   øʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *       NewState: ʹ״̬
  *                 ȡֵΧΪ
  *                   TRUE: 
  *                   FALSE: ֹ
  *   
  */
void
GPTIM_Updata_Enable (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴTXUDENλ --------------------*/
    if (NewState != FALSE)
    {
        /*  */
        SFR_SET_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUDEN_POS);
    }
    else
    {
        /* ÿڸ */
        SFR_CLR_BIT_ASM(GPTIMx->CTL2, GPTIM_CTL2_TXUDEN_POS);
    }
}

/**
  *   ¼DMA
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ¼DMAʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Trigger_DMA_Enable (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴTXTDEλ --------------------*/
    if (NewState != FALSE)
    {
        /* ¼DMA */
        SFR_SET_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXTDE_POS);
    }
    else
    {
        /* ֹ¼DMA */
        SFR_CLR_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXTDE_POS);
    }
}

/**
  *   ø¼DMAʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ¼DMA
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Updata_DMA_Enable (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴTXUDEλ --------------------*/
    if (NewState != FALSE)
    {
        /* ¼DMA */
        SFR_SET_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXUDE_POS);
    }
    else
    {
        /* ֹ¼DMA */
        SFR_CLR_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXUDE_POS);
    }
}

/**
  *   ȡTxжʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: Txжϣ
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Overflow_INT_Enable (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- ȡTXCTL1ĴTXIEλ --------------------*/
    if (NewState != FALSE)
    {
        /* Txж */
        SFR_SET_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXIE_POS);
    }
    else
    {
        /* ֹTxж */
        SFR_CLR_BIT_ASM(GPTIMx->CTL1, GPTIM_CTL1_TXIE_POS);
    }
}

/**
  *   Tx¼жʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: Tx¼жϣ
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Updata_INT_Enable (GPTIM_SFRmap* GPTIMx,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴTXUIEλ --------------------*/
    if (NewState != FALSE)
    {
        /* Tx¼ж */
        SFR_SET_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXUIE_POS);
    }
    else
    {
        /* ֹTx¼ж */
        SFR_CLR_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXUIE_POS);
    }
}

/**
  *   Tx¼жʹܡ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: Tx¼жϣ
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Trigger_INT_Enable (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴTXTIEλ --------------------*/
    if (NewState != FALSE)
    {
        /* Tx¼ж */
        SFR_SET_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXTIE_POS);
    }
    else
    {
        /* ֹTx¼ж */
        SFR_CLR_BIT_ASM(GPTIMx->CCPXCTL3, CCP_CTL3_TXTIE_POS);
    }
}

/**
  *   ¼λ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ʱʹܿ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
GPTIM_Generate_Trigger_Config (GPTIM_SFRmap* GPTIMx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXEGIFĴTXTRGλ --------------------*/
    if (NewState != FALSE)
    {
        /* Ӧ */
        SFR_SET_BIT_ASM(GPTIMx->CCPXEGIF, CCP_EGIF_TXTRG_POS);
    }
    else
    {
        /* ޶ */
        SFR_CLR_BIT_ASM(GPTIMx->CCPXEGIF, CCP_EGIF_TXTRG_POS);
    }
}

/**
  *   TX
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *   TX0£1ϡ
  */
DIRStatus
GPTIM_Get_Direction (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- ȡTXCTL1ĴTXDIRλ --------------------*/
    if (GPTIMx->CTL1 & GPTIM_CTL1_TXDIR)
    {
        /* ǰΪϼ */
        return DIR_UP;
    }
    else
    {
        /* ǰΪ¼ */
        return DIR_DOWN;
    }
}

/**
  *   ʱֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *   Txֵ32λ
  */
uint32_t
GPTIM_Get_Counter (GPTIM_SFRmap* GPTIMx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /* ʱֵTxCNT */
        tmpreg = GPTIMx->CNT;
    }
    else
    {
        /* ʱֵTxCNT */
        tmpreg = (uint16_t)GPTIMx->CNT;
    }
    return tmpreg;
}

/**
  *   ʱֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *   Txֵ32λ
  */
uint32_t
GPTIM_Get_Period (GPTIM_SFRmap* GPTIMx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /* ʱֵTxPPX */
        tmpreg = GPTIMx->PPX;
    }
    else
    {
        /* ʱֵTxPPX */
        tmpreg = (uint16_t)GPTIMx->PPX;
    }
    return tmpreg;
}

/**
  *   ʱԤƵֵ
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *   TxԤƵֵ32λ
  */
uint32_t
GPTIM_Get_Prescaler (GPTIM_SFRmap* GPTIMx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    if (GPTIMx == T21_SFR)
    {
        /* ʱԤƵֵTxPRSC */
        tmpreg = GPTIMx->PRSC;
    }
    else
    {
        /* ʱԤƵֵTxPRSC */
        tmpreg = (uint16_t)GPTIMx->PRSC;
    }
    return tmpreg;
}
/**
  *   ##### ͨöʱ(GPTIM)ʼú #####
  */


/**
  *   ##### ͨöʱ(GPTIM)жϼ־ #####
  */
/**
  *   Txжϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   
  */
void
GPTIM_Clear_Overflow_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- CCPXSRICĴTXICλ --------------------*/
    SFR_SET_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXIC_POS);
    while((GPTIMx->CTL1 & GPTIM_CTL1_TXIF)>>GPTIM_CTL1_TXIF_POS);
    SFR_CLR_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXIC_POS);
}

/**
  *   Tx¼жϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   
  * ˵ñ־λʹܶӦʱ½Уᵼʧܡ
  */
void
GPTIM_Clear_Updata_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- CCPXSRICĴTXUICλ --------------------*/
    SFR_SET_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXUIC_POS);
    while((GPTIMx->CCPXEGIF & CCP_EGIF_TXUIF)>>CCP_EGIF_TXUIF_POS);
    SFR_CLR_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXUIC_POS);
}

/**
  *   Tx¼жϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   
  */
void
GPTIM_Clear_Trigger_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- CCPXSRICĴTXTICλ --------------------*/
    SFR_SET_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXTIC_POS);
    while((GPTIMx->CCPXEGIF & CCP_EGIF_TXTIF)>>CCP_EGIF_TXTIF_POS);
    SFR_CLR_BIT_ASM(GPTIMx->CCPXSRIC, CCP_SRIC_TXTIC_POS);
}

/**
  *   ȡTxжϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   ж״̬0δжϣ1ж
  */
FlagStatus
GPTIM_Get_Overflow_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- ȡTXCTL1ĴTXIFλ --------------------*/
    if (GPTIMx->CTL1 & GPTIM_CTL1_TXIF)
    {
        /* Txж */
        return SET;
    }
    else
    {
        /* δTxж */
        return RESET;
    }
}

/**
  *   ȡTx¼жϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   ж״̬0δжϣ1ж
  */
FlagStatus
GPTIM_Get_Updata_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- ȡCCPXEGIFĴTXUIFλ --------------------*/
    if (GPTIMx->CCPXEGIF & CCP_EGIF_TXUIF)
    {
        /* Tx¼ж */
        return SET;
    }
    else
    {
        /* δTx¼ж */
        return RESET;
    }
}

/**
  *   ȡTx¼жϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *   ж״̬0δжϣ1ж
  */
FlagStatus
GPTIM_Get_Trigger_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- ȡCCPXEGIFĴTXTIFλ --------------------*/
    if (GPTIMx->CCPXEGIF & CCP_EGIF_TXTIF)
    {
        /* Tx¼ж */
        return SET;
    }
    else
    {
        /* δTx¼ж */
        return RESET;
    }
}

/**
  *   ȡTx¼DMAжϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   ж״̬0δжϣ1ж.DMA ӦλӲԶ
  */
FlagStatus
GPTIM_Get_Updata_DMA_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- CCPXDFĴTXUDFλ --------------------*/
    if (GPTIMx->CCPXDF & CCP_DF_TXUDF)
    {
        /* ¼DMAж */
        return SET;
    }
    else
    {
        /* δ¼DMAж */
        return RESET;
    }
}

/**
  *   ô¼DMAжϱ־
  *   GPTIMx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *   ж״̬0δжϣ1ж.DMA ӦλӲԶ
  */
FlagStatus
GPTIM_Get_Trigger_DMA_INT_Flag (GPTIM_SFRmap* GPTIMx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(GPTIMx));

    /*-------------------- CCPXDFĴTXTDFλ --------------------*/
    if (GPTIMx->CCPXDF & CCP_DF_TXTDF)
    {
        /* ¼DMAж */
        return SET;
    }
    else
    {
        /* δ¼DMAж */
        return RESET;
    }
}
/**
  *   ##### ͨöʱ(GPTIM)жϼ־ #####
  */


/**
  *   ##### ͨò׽/Ƚ/PWM(CCP)׽ܺ #####
  */
/**
  *   ͨñȽ(CCP)á
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       ccpInitStruct: ͨñȽ(CCP)Ϣṹָ롣
  *   
  */
void
CCP_Compare_Configuration(CCP_SFRmap* CCPx,
                    CCP_CompareInitTypeDef* ccpInitStruct)
{
    uint32_t tmpreg = 0;
    uint32_t tmpreg1 = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(ccpInitStruct->m_Channel));
    CHECK_RESTRICTION(CHECK_CCP_CMP_MODE(ccpInitStruct->m_CompareMode));

    /*------------------ CCPXCTL1Ĵ -----------------*/
    /* ݽṹԱm_Channelm_CompareModeCCPxCHyMλ */
    tmpreg = 4 * ccpInitStruct->m_Channel;
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << tmpreg),
                          (uint32_t)ccpInitStruct->m_CompareMode << tmpreg);

    /*------------------ CCPXRyĴ -----------------*/
    /* ݽṹԱm_CompareValueCCPXRyλ */
    /* ȡCCPXRYĴַ */
    tmpreg1 = (uint32_t)CCPx;
    tmpreg1 = tmpreg1 + CCP_RY_OFFSET + tmpreg;
    if (CCPx == CCP21_SFR)
    {
        *(volatile uint32_t*)tmpreg1
            = ccpInitStruct->m_CompareValue;
    }
    else
    {
        *(volatile uint32_t*)tmpreg1
            = (uint16_t)ccpInitStruct->m_CompareValue;
    }
}

/**
  *   ʼͨñȽ(CCP)Ϣṹ塣
  *   ccpInitStruct: ָʼĽṹָ롣
  *   
  */
void
CCP_Compare_Struct_Init (CCP_CompareInitTypeDef* ccpInitStruct)
{
    /* ñȽͨ */
    ccpInitStruct->m_Channel = CCP_CHANNEL_1;
    /* ͨģʽѡλ */
    ccpInitStruct->m_CompareMode = CCP_MODE_RST;
    /* ñȽֵ */
    ccpInitStruct->m_CompareValue = 0x00000000;
}

/**
  *   ͨò׽ģ(CCP)á
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       ccpInitStruct: ͨò׽ģ(CCP)Ϣṹָ롣
  *   
  */
void
CCP_Capture_Configuration (CCP_SFRmap* CCPx,
                    CCP_CaptureInitTypeDef* ccpInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(ccpInitStruct->m_Channel));
    CHECK_RESTRICTION(CHECK_CCP_CAP_MODE(ccpInitStruct->m_CaptureMode));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(ccpInitStruct->m_PWMInput));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(ccpInitStruct->m_XORMode));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(ccpInitStruct->m_ChannelCompare4));

    /*------------------ CCPXCTL1Ĵ -----------------*/
    /* ݽṹԱm_Channelm_CaptureModeCCPxCHyMλ */
    tmpreg = 4 * ccpInitStruct->m_Channel;
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << tmpreg),
                          (uint32_t)ccpInitStruct->m_CaptureMode << tmpreg);

    /*------------------ CCPXCTL2Ĵ -----------------*/
    /* ݽṹԱm_ChannelCompare4PxOC4CEλ */
    CCPx->CCPXCTL2 = SFR_Config (CCPx->CCPXCTL2,
                          ~CCP_CTL2_PXOC4CE,
                          ccpInitStruct->m_ChannelCompare4);

    /*------------------ CCPXCTL3Ĵ -----------------*/
    /* ݽṹԱm_PWMInputPxPWMIλ */
    /* ݽṹԱm_XORModePxXORMλ */
    tmpreg = (ccpInitStruct->m_PWMInput << CCP_CTL3_PXPWMI_POS) \
             | (ccpInitStruct->m_XORMode << CCP_CTL3_PXXORM_POS);
    CCPx->CCPXCTL3 = SFR_Config (CCPx->CCPXCTL3,
                          ~CCP_CTL3_CAP_INIT_MASK,
                          tmpreg);
}

/**
  *   ʼͨò׽ģ(CCP)Ϣṹ塣
  *   ccpInitStruct: ָʼĽṹָ롣
  *   
  */
void
CCP_Capture_Struct_Init (CCP_CaptureInitTypeDef* ccpInitStruct)
{
    /* ò׽ͨ */
    ccpInitStruct->m_Channel = CCP_CHANNEL_1;
    /* ͨģʽѡλ */
    ccpInitStruct->m_CaptureMode = CCP_MODE_RST;
    /* PWMģʽ */
    ccpInitStruct->m_PWMInput = FALSE;
    /* ģʽ */
    ccpInitStruct->m_XORMode = FALSE;
    /* ñȽ׽ */
    ccpInitStruct->m_ChannelCompare4 = FALSE;
}

/**
  *   ͨPWMģ(CCP)á
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       ccpInitStruct: ͨPWMģ(CCP)Ϣṹָ롣
  *   
  */
void
CCP_PWM_Configuration (CCP_SFRmap* CCPx,
                    CCP_PWMInitTypeDef* ccpInitStruct)
{
    uint32_t tmpreg = 0;
    uint32_t tmpreg1 = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(ccpInitStruct->m_Channel));
    CHECK_RESTRICTION(CHECK_CCP_PWM_MODE(ccpInitStruct->m_PwmMode));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL_OUTPUT(ccpInitStruct->m_OutputCtl));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(ccpInitStruct->m_SinglePWM));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(ccpInitStruct->m_CloseTimer));

    /*------------------ CCPXCTL1Ĵ -----------------*/
    /* ݽṹԱm_Channelm_PwmModeCCPxCHyMλ */
    tmpreg = 4 * ccpInitStruct->m_Channel;
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << tmpreg),
                          (uint32_t)ccpInitStruct->m_PwmMode << tmpreg);

    /*------------------ CCPXRyĴ -----------------*/
    /* ݽṹԱm_ChannelȡCCPXRYĴַ */
    tmpreg1 = (uint32_t)CCPx;
    tmpreg1 = tmpreg1 + CCP_RY_OFFSET + tmpreg;
    /* ݽṹԱm_DutyRatioCCPXRyλ */
    if (CCPx == CCP21_SFR)
    {
        *(volatile uint32_t*)tmpreg1
            = ccpInitStruct->m_DutyRatio;
    }
    else
    {
        *(volatile uint32_t*)tmpreg1
            = (uint16_t)ccpInitStruct->m_DutyRatio;
    }

    /*------------------ CCPXCTL2Ĵ -----------------*/
    /* ݽṹԱm_Channelm_OutputCtlPxOCyλ */
    tmpreg = 2 * ccpInitStruct->m_Channel;
    CCPx->CCPXCTL2 = SFR_Config (CCPx->CCPXCTL2,
                          ~(CCP_CTL2_PXOC1 << tmpreg),
                          ccpInitStruct->m_OutputCtl << tmpreg);

    /*------------------ CCPXCTL3Ĵ -----------------*/
    /* ݽṹԱm_SinglePWMPxSPMλ */
    /* ݽṹԱm_CloseTimerPxSPMSTλ */
    tmpreg = (ccpInitStruct->m_SinglePWM << CCP_CTL3_PXSPM_POS) \
             | (ccpInitStruct->m_CloseTimer << CCP_CTL3_PXSPMST_POS);
    CCPx->CCPXCTL3 = SFR_Config (CCPx->CCPXCTL3,
                          ~CCP_CTL3_PWM_INIT_MASK,
                          tmpreg);
}

/**
  *   ʼͨPWMģ(CCP)Ϣṹ塣
  *   ccpInitStruct: ָʼĽṹָ롣
  *   
  */
void
CCP_PWM_Struct_Init (CCP_PWMInitTypeDef* ccpInitStruct)
{
    /* PWMͨ */
    ccpInitStruct->m_Channel = CCP_CHANNEL_1;
    /* ͨģʽѡλ */
    ccpInitStruct->m_PwmMode = CCP_MODE_RST;
    /* PWMռձ */
    ccpInitStruct->m_DutyRatio = 0x00000000;
    /*  */
    ccpInitStruct->m_OutputCtl = CCP_CHANNEL_OUTPUT_PWM_ACTIVE;
    /* õģʽ */
    ccpInitStruct->m_SinglePWM = FALSE;
    /* õģʽѡ */
    ccpInitStruct->m_CloseTimer = FALSE;
}

/**
  *   CCP׽ͨģʽ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       EdgeConfig: ƥأȡֵΧΪ:
  *                     CCP_MODE_RST: ģʽر
  *                     CCP_CAP_FALLING_EDGE: ÿ½ط׽
  *                     CCP_CAP_RISING_EDGE: ÿط׽
  *                     CCP_CAP_4TH_RISING_EDGE: ÿ4ط׽
  *                     CCP_CAP_16TH_RISING_EDGE: ÿ16ط׽
  *   
  */
void
CCP_Capture_Mode_Config (CCP_SFRmap* CCPx, uint32_t Channel, uint32_t EdgeConfig)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_CCP_CAP_MODE(EdgeConfig));

    /*-------------------- CCPXCTL1Ĵ --------------------*/
    tmpreg = EdgeConfig << (4 * Channel);
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << (4 * Channel)),
                          tmpreg);
}

/**
  *   CCPȽϹͨģʽ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       EdgeConfig: ƥأȡֵΧΪ:
  *                     CCP_MODE_RST: ģʽر
  *                     CCP_CMP_ACTIVE_LEVEL: Ƚƥʱߵƽ
  *                     CCP_CMP_INACTIVE_LEVEL: Ƚƥʱ͵ƽ
  *                     CCP_CMP_SOFT_INT: Ƚƥʱж
  *                     CCP_CMP_SPECIAL_EVENT: ¼
  *   
  */
void
CCP_Compare_Mode_Config (CCP_SFRmap* CCPx, uint32_t Channel, uint32_t EdgeConfig)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_CCP_CMP_MODE(EdgeConfig));

    /*-------------------- CCPXCTL1Ĵ --------------------*/
    tmpreg = EdgeConfig << (4 * Channel);
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << (4 * Channel)),
                          tmpreg);
}

/**
  *   CCP PWMͨģʽ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       EdgeConfig: ƥأȡֵΧΪ:
  *                     CCP_MODE_RST: ģʽر
  *                     CCP_PWM_MODE: PWMģʽ
  *   
  */
void
CCP_PWM_Mode_Config (CCP_SFRmap* CCPx, uint32_t Channel, uint32_t EdgeConfig)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_CCP_PWM_MODE(EdgeConfig));

    /*-------------------- CCPXCTL1Ĵ --------------------*/
    tmpreg = EdgeConfig << (4 * Channel);
    CCPx->CCPXCTL1 = SFR_Config (CCPx->CCPXCTL1,
                          ~(CCP_CTL1_CH1M << (4 * Channel)),
                          tmpreg);
}

/**
  *   ȡCCP׽Ĵ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   ׽Ĵֵ
  */
uint32_t
CCP_Get_Capture_Result (CCP_SFRmap* CCPx, uint32_t Channel)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));

    /* ȡĴʵַ */
    tmpreg = (uint32_t)CCPx;
    tmpreg = tmpreg + CCP_CY_OFFSET + (4 * Channel);

    /* ȡĴֵ */
    if (CCPx == CCP21_SFR)
    {
        return (*(volatile const uint32_t*)  tmpreg);
    }
    else
    {
        return (uint16_t)(*(volatile const uint32_t*)  tmpreg);
    }
}

/**
  *   CCPȽ/PWMռձȼĴ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   ޡ
  */
void
CCP_Set_Compare_Result (CCP_SFRmap* CCPx, uint32_t Channel, uint32_t Value)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));

    /* ȡCCPXRYĴַ */
    tmpreg = (uint32_t)CCPx;
    tmpreg = tmpreg + CCP_RY_OFFSET + (4 * Channel);

    /*-------------------- CCPXRYĴ --------------------*/
    if (CCPx == CCP21_SFR)
    {
        *(volatile uint32_t*)tmpreg = Value;
    }
    else
    {
        *(volatile uint32_t*)tmpreg = (uint16_t)Value;
    }
}

/**
  *   ȡCCPȽ/PWMռձȼĴ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   Ƚ/PWMռձȼĴֵ
  */
uint32_t
CCP_Get_Compare_Result (CCP_SFRmap* CCPx, uint32_t Channel)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));

    /* ȡĴʵַ */
    tmpreg = (uint32_t)CCPx;
    tmpreg = tmpreg + CCP_RY_OFFSET + (4 * Channel);

    /* ȡĴֵ */
    if (CCPx == CCP21_SFR)
    {
        return (*(volatile uint32_t*)  tmpreg);
    }
    else
    {
        return (uint16_t)(*(volatile uint32_t*)  tmpreg);
    }
}

/**
  *   CCPȽ/PWMźʹλ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       NewState: CCPȽ/PWMźʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   ޡ
  */
void
CCP_Compare_PWM_Signal_Clear (CCP_SFRmap* CCPx, uint32_t Channel,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL2ĴPXOCyCEλ --------------------*/
    CCPx->CCPXCTL2 = SFR_Config (CCPx->CCPXCTL2,
                          ~(CCP_CTL2_PXOC1CE << (Channel)),
                          (uint32_t)NewState << (CCP_CTL2_PXOC1CE_POS + Channel));
}

/**
  *   CCPȽѡ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       CompareSelect: ȽѡȡֵΧΪ:
  *                        CCP_COMPARE_SELECT_1: Ƚ1
  *                        CCP_COMPARE_SELECT_2: Ƚ2
  *                        CCP_COMPARE_SELECT_3: Ƚ3
  *   ޡ
  */
void
CCP_Compare_Check_Config (CCP_SFRmap* CCPx, uint32_t CompareSelect)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_COMPARE_SELECT(CompareSelect));

    /*-------------------- CCPXCTL2ĴPXASSλ --------------------*/
    CCPx->CCPXCTL2 = SFR_Config (CCPx->CCPXCTL2,
                          ~(CCP_CTL2_PXASS),
                          CompareSelect);
}

/**
  *   CCPͨơ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ׽ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       ChannelOutputCtl: ȽѡȡֵΧΪ:
  *                           CCP_CHANNEL_OUTPUT_PWM_ACTIVE: PWMЧ
  *                           CCP_CHANNEL_OUTPUT_PWM_INACTIVE: PWMЧ
  *                           CCP_CHANNEL_OUTPUT_INACTIVE: ǿƵ͵ƽ
  *                           CCP_CHANNEL_OUTPUT_ACTIVE: ǿƸߵƽ
  *   ޡ
  */
void
CCP_Channel_Output_Control (CCP_SFRmap* CCPx, uint32_t Channel,
                    uint32_t ChannelOutputCtl)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL_OUTPUT(ChannelOutputCtl));

    /*-------------------- CCPXCTL2ĴPXOCyλ --------------------*/
    tmpreg = ChannelOutputCtl << (Channel * 2);
    CCPx->CCPXCTL2 = SFR_Config (CCPx->CCPXCTL2,
                          ~(CCP_CTL2_PXOC1 << (Channel * 2)),
                          tmpreg);
}

/**
  *   õģʽ
  *   CCPx: ָʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ģʽѡ
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_Single_Pulse_Shut_Enable (CCP_SFRmap* CCPx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴPXSPMSTλ --------------------*/
    if (NewState != FALSE)
    {
        /* رնʱʹλ */
        SFR_SET_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXSPMST_POS);
    }
    else
    {
        /* 󲻹رնʱʹλ */
        SFR_CLR_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXSPMST_POS);
    }
}

/**
  *   õģʽ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ģʽʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_Single_Pulse_Enable (CCP_SFRmap* CCPx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴPXSPMλ --------------------*/
    if (NewState != FALSE)
    {
        /* ģʽʹ */
        SFR_SET_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXSPM_POS);
    }
    else
    {
        /* ģʽδʹ */
        SFR_CLR_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXSPM_POS);
    }
}

/**
  *   PWMģʽʹ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: PWMģʽʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_PWM_Input_Measurement_Config (CCP_SFRmap* CCPx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴPXPWMIλ --------------------*/
    if (NewState != FALSE)
    {
        /* ʹPWMģʽ */
        SFR_SET_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXPWMI_POS);
    }
    else
    {
        /* δʹPWMģʽ */
        SFR_CLR_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXPWMI_POS);
    }
}

/**
  *   ʹλ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_Input_XOR_Config (CCP_SFRmap* CCPx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- TXCTL2ĴPXXORMλ --------------------*/
    if (NewState != FALSE)
    {
        /* ʹPWMģʽ */
        SFR_SET_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXXORM_POS);
    }
    else
    {
        /* δʹPWMģʽ */
        SFR_CLR_BIT_ASM(CCPx->CCPXCTL3, CCP_CTL3_PXXORM_POS);
    }
}

/**
  *   /ȽͨDMAá
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       NewState: /ȽͨDMAʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   ޡ
  */
void
CCP_Channel_DMA_Config (CCP_SFRmap* CCPx, uint32_t Channel,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴCCPXCCYDEλ --------------------*/
    if (NewState != FALSE)
    {
        /* /ȽͨDMA */
        CCPx->CCPXCTL3 |= CCP_CTL3_CC1DE << Channel;
    }
    else
    {
        /* ֹ/ȽͨDMA */
        CCPx->CCPXCTL3 &= ~(CCP_CTL3_CC1DE << Channel);
    }
}

/**
  *   ׽/Ƚ¼λ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       NewState: ׽/Ƚ¼״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_Generate_Trigger_Config (CCP_SFRmap* CCPx, uint32_t Channel,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*------------------ CCPXEGIFĴCCPXCCyGλ ------------------*/
    if (NewState != FALSE)
    {
        /* ͨChannelϲһ׽/Ƚ¼ */
        CCPx->CCPXEGIF |= CCP_EGIF_CC1G << Channel;
    }
    else
    {
        /* ޶ */
        CCPx->CCPXEGIF &= ~(CCP_EGIF_CC1G << Channel);
    }
}

/**
  *   ڽźŲ¼λ
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       NewState: ڽźŲʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   
  */
void
CCP_USART_Receive_Config (CCP_SFRmap* CCPx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXEGIFĴUSARTRXENλ --------------------*/
    if (NewState != FALSE)
    {
        /* ʹܲڽź */
        SFR_SET_BIT_ASM(CCPx->CCPXEGIF, CCP_EGIF_UARTRXEN_POS);
    }
    else
    {
        /* ֹڽź */
        SFR_CLR_BIT_ASM(CCPx->CCPXEGIF, CCP_EGIF_UARTRXEN_POS);
    }
}
/**
  *   ##### ͨò׽/Ƚ/PWM(CCP)׽ܺ #####
  */


/**
  *   ##### ͨò׽/Ƚ/PWM(CCP)жϼ־ #####
  */
/**
  *   ȡCCPͨ׽/Ƚжϱ־
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   ж״̬0δжϣ1ж
  */
FlagStatus
CCP_Get_Channel_Trigger_INT_Flag (CCP_SFRmap* CCPx, uint32_t Channel)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));

    /*-------------------- ȡCCPXEGIFĴTXTIFλ --------------------*/
    tmpreg = CCP_EGIF_CC1IF << Channel;
    if (CCPx->CCPXEGIF & tmpreg)
    {
        /* ж */
        return SET;
    }
    else
    {
        /* δж */
        return RESET;
    }
}

/**
  *   ȡCCPͨDMAжϱ־
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   ж״̬0δжϣ1ж.DMA ӦλӲԶ
  */
FlagStatus
CCP_Get_Trigger_DMA_INT_Flag (CCP_SFRmap* CCPx, uint32_t Channel)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));

    /*-------------------- ȡCCPXDFĴCCyDFλ --------------------*/
    tmpreg = CCP_DF_CC1DF << Channel;
    if (CCPx->CCPXDF & tmpreg)
    {
        /*  CCPͨDMA */
        return SET;
    }
    else
    {
        /*  CCPͨ޴DMA */
        return RESET;
    }
}

/**
  *   ͨ/Ƚϵжʹá
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *       NewState: ͨ/Ƚϵжʹ״̬
  *                 ȡֵΧΪTRUE  FALSE
  *   ޡ
  */
void
CCP_Channel_INT_Config (CCP_SFRmap* CCPx, uint32_t Channel,
                    FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*-------------------- CCPXCTL3ĴCCPXCCYIEλ --------------------*/
    if (NewState != FALSE)
    {
        /* CCPͨChannel׽/Ƚж */
        CCPx->CCPXCTL3 |= CCP_CTL3_CC1IE << Channel;
    }
    else
    {
        /* ֹCCPͨChannel׽/Ƚж */
        CCPx->CCPXCTL3 &= ~(CCP_CTL3_CC1IE << Channel);
    }
}

/**
  *   CCPͨ/Ƚϵжϱ־
  *   CCPx: ָCCPͨöʱڴṹָ룬
  *               ȡֵT0_SFR/T1_SFR/T2_SFR/T3_SFR/T4_SFR/T18_SFR/
  *               T19_SFR/T21_SFR
  *               CCP0_SFR/CCP1_SFR/CCP2_SFR/CCP3_SFR/CCP4_SFR/
  *               CCP18_SFR/CCP19_SFR/CCP21_SFR
  *       Channel: ͨѡȡֵΧΪ:
  *                  CCP_CHANNEL_1: ͨ1
  *                  CCP_CHANNEL_2: ͨ2
  *                  CCP_CHANNEL_3: ͨ3
  *                  CCP_CHANNEL_4: ͨ4
  *   ޡ
  */
void
CCP_Clear_Channel_INT_Flag (CCP_SFRmap* CCPx, uint32_t Channel)
{
	uint32_t tmpreg = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_GPTIM_PERIPH(CCPx));
    CHECK_RESTRICTION(CHECK_CCP_CHANNEL(Channel));

    /*-------------------- CCPXSRICĴCCPXCCYICλ --------------------*/
    tmpreg = CCP_EGIF_CC1IF << Channel;

    CCPx->CCPXSRIC |= CCP_SRIC_CC1IC << Channel;
    while((CCPx->CCPXEGIF & tmpreg)>>(Channel+CCP_EGIF_CC1IF_POS));
    CCPx->CCPXSRIC &= ~(CCP_SRIC_CC1IC << Channel);
}
/**
  *   ##### ͨò׽/Ƚ/PWM(CCP)жϼ־ #####
  */
