/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : kf32a156_can.c                                 	      *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This file provides functions related to the CAN bus   *
 *  					bus (CAN) , including:								  *
 *          			+ CAN Bus(CAN)initialization function                 *
 *          			+ CAN Bus(CAN)function configuration function         *
 *          			+ CAN Bus(CAN)transmit-receive function               *
 *          			+ CAN Bus(CAN)interrupt management function	          *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/

/******************************************************************************
**                      		Include Files                                **
******************************************************************************/

#include "kf32a156_can.h"
#include "kf32a156_rst.h"
#include "kf32a156_pclk.h"

/**
  *   ȳʼRAMRAMȫ
  */
void CAN_RAM_ERASE(void)
{
	for(uint16_t i = 0;i< 1024;i++)
	{
		*((uint8_t *)CAN4_RECEIVE_ADDR + i) = 0x00;
	}
}

/**
  *   ##### (CAN)ʼ #####
  */
/**
  *   λCAN裬ʹʱӡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ޡ
  */
void
CAN_Reset(CAN_SFRmap* CANx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    if(CANx == CAN4_SFR)
    {
        RST_CTL3_Peripheral_Reset_Enable(RST_CTL3_CAN4RST, TRUE);
        PCLK_CTL3_Peripheral_Clock_Enable(PCLK_CTL3_CAN4CLKEN, TRUE);
        RST_CTL3_Peripheral_Reset_Enable(RST_CTL3_CAN4RST, FALSE);
    }
    else
    {
        ;
    }
}

/**
  *   (CAN)ʼúһλģʽRSMOD
  *       ںʱRSMOD
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       canInitStruct: CANϢṹ塣
  *   ޡ
  */
void
CAN_Configuration_With_Reset_Mode(CAN_SFRmap* CANx, CAN_InitTypeDef* canInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(canInitStruct->m_Enable));
    CHECK_RESTRICTION(CHECK_CAN_WORK_MODE(canInitStruct->m_Mode));
    CHECK_RESTRICTION(CHECK_CAN_SOURCE(canInitStruct->m_WorkSource));
    CHECK_RESTRICTION(CHECK_CAN_BAUDRATE_PRESET(canInitStruct->m_BaudRate));
    CHECK_RESTRICTION(CHECK_CAN_SYNC_JMP_WIDTH(canInitStruct->m_SyncJumpWidth));
    CHECK_RESTRICTION(CHECK_CAN_TIME_SEGMENT1(canInitStruct->m_TimeSeg1));
    CHECK_RESTRICTION(CHECK_CAN_TIME_SEGMENT2(canInitStruct->m_TimeSeg2));
    CHECK_RESTRICTION(CHECK_CAN_BUS_SAMPLE(canInitStruct->m_BusSample));

    /*---------------- CANx_CTLRĴ ----------------*/
    /* ݽṹԱm_EnableCANENλ */
    /* ݽṹԱm_ModeLBACKSILENTλ */
    /* ݽṹԱm_WorkSourceCANCKSλ */
    /* RSMOD븴λģʽ */
    tmpreg = ((uint32_t)canInitStruct->m_Enable << CAN_CTLR_CANEN_POS) \
           | canInitStruct->m_Mode \
           | canInitStruct->m_WorkSource \
           | CAN_CTLR_RSMOD;
    CANx->CTLR = SFR_Config (CANx->CTLR, ~CAN_CTLR_INIT_MASK, tmpreg);

    /*---------------- CANx_BRGRĴ ----------------*/
    /* ݽṹԱm_BaudRateCANBRPλ */
    /* ݽṹԱm_SyncJumpWidthSJWλ */
    /* ݽṹԱm_TimeSeg1TSEG1λ */
    /* ݽṹԱm_TimeSeg2TSEG2λ */
    /* ݽṹԱm_BusSampleSAMλ */
    tmpreg = ((uint32_t)canInitStruct->m_BaudRate << CAN_BRGR_CANBRP0_POS) \
           | ((uint32_t)canInitStruct->m_SyncJumpWidth << CAN_BRGR_SJW0_POS) \
           | ((uint32_t)canInitStruct->m_TimeSeg1 << CAN_BRGR_TSEG1_0_POS) \
           | ((uint32_t)canInitStruct->m_TimeSeg2 << CAN_BRGR_TSEG2_0_POS) \
           | (canInitStruct->m_BusSample);
    CANx->BRGR = SFR_Config (CANx->BRGR, ~CAN_BRGR_INIT_MASK, tmpreg);
}

/**
  *   ʼCANϢṹ塣
  *   canInitStruct: ָʼĽṹָ롣
  *   ޡ
  */
void
CAN_Struct_Init (CAN_InitTypeDef* canInitStruct)
{
    /* ʼ CANʹѡ */
    canInitStruct->m_Enable = FALSE;

    /* ʼ CANģʽ */
    canInitStruct->m_Mode = CAN_MODE_NORMAL;

    /* ʼ CANʱ */
    canInitStruct->m_WorkSource = CAN_SOURCE_SCLK_DIV_2;

    /* ʼ CANԤֵ */
    canInitStruct->m_BaudRate = 0;

    /* ʼ CANͬת */
    canInitStruct->m_SyncJumpWidth = 0;

    /* ʼ CANʱ1 */
    canInitStruct->m_TimeSeg1 = 0;

    /* ʼ CANʱ2 */
    canInitStruct->m_TimeSeg2 = 0;

    /* ʼ CAN߲ */
    canInitStruct->m_BusSample = CAN_BUS_SAMPLE_1_TIME;
}
/**
  *   ##### (CAN)ʼ #####
  */


/**
  *   ##### (CAN)ú #####
  */
/**
  *   ȡCAN RXϢ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   RXϢЧֵΪ7λ
  */
uint32_t
CAN_Get_Receive_Message_Counter (CAN_SFRmap* CANx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*---------------- ȡCANx_CTLRĴCANRMCλ ----------------*/
    tmpreg = CANx->CTLR;
    tmpreg &= CAN_CTLR_CANRMC;
    tmpreg >>= CAN_CTLR_CANRMC0_POS;

    return tmpreg;
}

/**
  *   ȡCAN¼״̬͡
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Type: CAN¼״̬ͣȡֵΪ:
  *               CAN_BUS_OFF_STATUS: ߹ر״̬
  *               CAN_ERROR_STATUS: ״̬
  *               CAN_TX_STATUS: ״̬
  *               CAN_RX_STATUS: ״̬
  *               CAN_TX_COMPLETE_OFF_STATUS: ״̬
  *               CAN_TX_BUFFER_STATUS: ͻ״̬
  *               CAN_DATA_OVERFLOW_STATUS: ״̬
  *               CAN_RX_BLANK_STATUS: ݿ״̬
  *   1: ߹ر//ڷ/ڽ//CPUԷʷͻ/
  *          RAMδ/RAMδϢ
  *       0: ߿/޴/Ϳ/տ/δ/CPUܷʷͻ/
  *          RAMδδ/RAMδϢ
  */
FlagStatus
CAN_Get_Transmit_Status (CAN_SFRmap* CANx, uint32_t Type)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_STATUS(Type));

    /*---------------- ȡCANx_CTLRĴӦλ ----------------*/
    tmpreg = CANx->CTLR;

    if (tmpreg & Type)
    {
        /* ״̬λΪ1 */
        return SET;
    }
    else
    {
        /* ״̬λΪ0 */
        return RESET;
    }
}

/**
  *   CANʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       NewState: CANʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
CAN_Cmd (CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- CANx_CTLRĴCANENλ ----------------*/
    if (NewState != FALSE)
    {
        /* CANʹ */
        SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_CANEN_POS);
    }
    else
    {
        /* CANֹ */
        SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_CANEN_POS);
    }
}

/**
  *   CANʱѡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       ClockSource: CANʱӣȡֵΪ:
  *                      CAN_SOURCE_SCLK_DIV_2: ѡSCLKʱ/2ΪCANʱ
  *                      CAN_SOURCE_HFCLK_DIV_2: ѡHFCLKʱ/2ΪCANʱ
  *                      CAN_SOURCE_LFCLK_DIV_2: ѡLFCLKʱ/2ΪCANʱ
  *   ޡ
  */
void
CAN_Clock_Source_Config (CAN_SFRmap* CANx, uint32_t ClockSource)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_SOURCE(ClockSource));

    /*---------------- CANx_CTLRĴCANCKSλ ----------------*/
    CANx->CTLR = SFR_Config (CANx->CTLR, ~CAN_CTLR_CANCKS, ClockSource);
}

/**
  *   CAN˯ģʽʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       NewState: CAN˯ģʽʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
CAN_Sleep_Mode_Enable (CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- CANx_CTLRĴSLEEPλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹ˯ģʽ */
        SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_SLEEP_POS);
    }
    else
    {
        /* ֹ˯ģʽ */
        SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_SLEEP_POS);
    }
}

/**
  *   CANλģʽʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       NewState: CANλģʽʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
CAN_Reset_Mode_Enable (CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- CANx_CTLRĴRSMODλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܸλģʽ */
        SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_RSMOD_POS);
    }
    else
    {
        /* ֹλģʽ */
        SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_RSMOD_POS);
    }
}

/**
  *   CANģʽ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       ModeType: CANģʽȡֵΪ:
  *                   CAN_MODE_NORMAL: ֹͻػģʽ
  *                   CAN_MODE_SILENT: ʹܰģʽ
  *                   CAN_MODE_LOOPBACK: ʹܻػģʽ
  *                   CAN_MODE_SILENT_LOOPBACK: ʹܰͻػģʽ
  *   ޡ
  */
void
CAN_Work_Mode_Config (CAN_SFRmap* CANx, uint32_t ModeType)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_WORK_MODE(ModeType));

    /*------------ CANx_CTLRĴSILENTLBACKλ ------------*/
    CANx->CTLR = SFR_Config (CANx->CTLR,
                        ~(CAN_CTLR_SILENT | CAN_CTLR_LBACK),
                        ModeType);
}

/**
  *   CAN߲
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Times: ߲ȡֵΪ:
  *                CAN_BUS_SAMPLE_1_TIME: ߲1
  *                CAN_BUS_SAMPLE_3_TIMES: ߲3
  *   ޡ
  */
void
CAN_Bus_Sample_Times_Config (CAN_SFRmap* CANx, uint32_t Times)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_BUS_SAMPLE(Times));

    /*---------------- CANx_BRGRĴSAMλ ----------------*/
    if (Times != CAN_BUS_SAMPLE_1_TIME)
    {
        /* ߲3 */
        SFR_SET_BIT_ASM(CANx->BRGR, CAN_BRGR_SAM_POS);
    }
    else
    {
        /* ߲1 */
        SFR_CLR_BIT_ASM(CANx->BRGR, CAN_BRGR_SAM_POS);
    }
}

/**
  *   CANʱΡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       TimeSeg1: CANʱ1ȡֵΪ4λЧֵ
  *       TimeSeg2: CANʱ2ȡֵΪ3λЧֵ
  *   ޡ
  */
void
CAN_Time_Segment_Config (CAN_SFRmap* CANx,
                    uint32_t TimeSeg1, uint32_t TimeSeg2)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_TIME_SEGMENT1(TimeSeg1));
    CHECK_RESTRICTION(CHECK_CAN_TIME_SEGMENT2(TimeSeg2));

    /*------------- CANx_BRGRĴTSEG1/TSEG2λ -------------*/
    tmpreg = (TimeSeg1 << CAN_BRGR_TSEG1_0_POS) \
           | (TimeSeg2 << CAN_BRGR_TSEG2_0_POS);
    CANx->BRGR = SFR_Config (CANx->BRGR,
                        ~(CAN_BRGR_TSEG1 | CAN_BRGR_TSEG2),
                        tmpreg);
}

/**
  *   CANͬתȡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       JumpWidth: ͬתȣȡֵΪ2λЧֵ
  *   ޡ
  */
void
CAN_Sync_Jump_Width_Config (CAN_SFRmap* CANx, uint32_t JumpWidth)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_SYNC_JMP_WIDTH(JumpWidth));

    /*------------- CANx_BRGRĴSJWλ -------------*/
    tmpreg = JumpWidth << CAN_BRGR_SJW0_POS;
    CANx->BRGR = SFR_Config (CANx->BRGR, ~CAN_BRGR_SJW, tmpreg);
}

/**
  *   CANԤֵ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       BaudRate: ԤֵȡֵΪ6λЧֵ
  *   ޡ
  */
void
CAN_Baud_Rate_Preset_Config (CAN_SFRmap* CANx, uint32_t BaudRate)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_BAUDRATE_PRESET(BaudRate));

    /*------------- CANx_BRGRĴCANBRPλ -------------*/
    tmpreg = BaudRate << CAN_BRGR_CANBRP0_POS;
    CANx->BRGR = SFR_Config (CANx->BRGR, ~CAN_BRGR_CANBRP, tmpreg);
}

/**
  *   ȡCANָ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ָ룬ȡֵΪ0~63
  */
uint8_t CAN_Get_Point_Of_RAM_Mailbox(CAN_SFRmap* CANx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    tmpreg = (CANx->RCR & CAN_RCR_RAMMB0) >> CAN_RCR_RAMMB0_POS;

    return (uint8_t)tmpreg;
}

/**
  *   ȡCANߴϢ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       canErrorStruct: CANߴϢṹ壬¼Ϣ
  *   ޡ
  */
void
CAN_Get_Error_Code (CAN_SFRmap* CANx, CAN_ErrorTypeDef* canErrorStruct)
{
    uint32_t tmpreg = 0;
    uint32_t tmpreg1 = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(canErrorStruct != (CAN_ErrorTypeDef*)0);

    /*---------------- ȡCANx_RCRĴ ----------------*/
    tmpreg = CANx->RCR;

    /*---------------- ȡCANx_RCRĴCANERRCλ ----------------*/
    tmpreg1 = tmpreg & CAN_RCR_CANERRC;
    tmpreg1 >>= CAN_RCR_CANERRC0_POS;
    canErrorStruct->m_ErrorCode = tmpreg1;

    /*---------------- ȡCANx_RCRĴCANDIRλ ----------------*/
    tmpreg1 = tmpreg & CAN_RCR_CANDIR;
    tmpreg1 >>= CAN_RCR_CANDIR_POS;
    canErrorStruct->m_ErrorDirection = tmpreg1;

    /*---------------- ȡCANx_RCRĴCANSEGλ ----------------*/
    tmpreg1 = tmpreg & CAN_RCR_CANSEG;
    tmpreg1 >>= CAN_RCR_CANSEG0_POS;
    canErrorStruct->m_ErrorSegment = tmpreg1;

    /*---------------- ȡCANx_RCRĴCANALCλ ----------------*/
    tmpreg1 = tmpreg & CAN_RCR_CANALC;
    tmpreg1 >>= CAN_RCR_CANALC0_POS;
    canErrorStruct->m_ArbitrationLost = tmpreg1;
}

/**
  *   ȡCAN󱨾ơ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   󱨾ƣȡֵΪ8λֵ
  */
uint8_t
CAN_Get_Error_Warning_Limit (CAN_SFRmap* CANx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*---------------- ȡCANx_ERORĴCANEWLλ ----------------*/
    tmpreg = CANx->EROR;

    return (uint8_t)tmpreg;
}

/**
  *   ȡCAN/մ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Direction: CAN䷽ȡֵΪ
  *                    CAN_ERROR_AT_TX: ʹ
  *                    CAN_ERROR_AT_RX: մ
  *   ȡֵΪ8λֵ
  */
uint8_t
CAN_Get_Error_Counter (CAN_SFRmap* CANx, uint32_t Direction)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ERROR_DIR(Direction));

    /*----------- ȡCANx_ERORĴ -----------*/
    tmpreg = CANx->EROR;

    if (Direction != CAN_ERROR_AT_TX)
    {
        /*----------- ȡCANRXEλ -----------*/
        tmpreg &= CAN_EROR_CANRXE;
        tmpreg >>= CAN_EROR_CANRXE0_POS;
    }
    else
    {
        /*----------- ȡCANTXEλ -----------*/
        tmpreg &= CAN_EROR_CANTXE;
        tmpreg >>= CAN_EROR_CANTXE0_POS;
    }

    return (uint8_t)tmpreg;
}

/**
  *   CAN󱨾ơ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       ErrorLimit: 󱨾ƣȡֵΪ8λЧֵ
  *   ޡ
  */
void
CAN_Error_Warning_Limit_Config (CAN_SFRmap* CANx, uint8_t ErrorLimit)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- CANx_ERORĴCANEWLλ -------------*/
    tmpreg = ErrorLimit << CAN_EROR_CANEWL0_POS;
    CANx->EROR = SFR_Config (CANx->EROR, ~CAN_EROR_CANEWL, tmpreg);
}

/**
  *   CAN/մ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Direction: CAN䷽ȡֵΪ
  *                    CAN_ERROR_AT_TX: ʹ
  *                    CAN_ERROR_AT_RX: մ
  *       ErrorCounter: ȡֵΪ8λЧֵ
  *   ޡ
  */
void
CAN_Error_Counter_Config (CAN_SFRmap* CANx,
                    uint32_t Direction, uint8_t ErrorCounter)
{
    uint32_t tmpreg = 0;
    uint32_t tmpmask = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ERROR_DIR(Direction));

    if (Direction != CAN_ERROR_AT_TX)
    {
        /*----------- ȡCANRXEλ -----------*/
        tmpreg = ErrorCounter << CAN_EROR_CANRXE0_POS;
        tmpmask = CAN_EROR_CANRXE;
    }
    else
    {
        /*----------- ȡCANTXEλ -----------*/
        tmpreg = ErrorCounter << CAN_EROR_CANTXE0_POS;
        tmpmask = CAN_EROR_CANTXE;
    }

    /*------------- CANx_ERORĴ -------------*/
    CANx->EROR = SFR_Config (CANx->EROR, ~tmpmask, tmpreg);
}

/**
  *   CANմ롣
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Acceptance: մ룬ȡֵΪ32λЧֵ
  *   ޡ
  */
void
CAN_Acceptance_Config (CAN_SFRmap* CANx, uint32_t Acceptance)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- CANx_ACRRĴACRλ -------------*/
    CANx->ACRR = Acceptance;
}

/**
  *   ȡCANմ롣
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   մ룬ȡֵΪ32λЧֵ
  */
uint32_t
CAN_Get_Acceptance (CAN_SFRmap* CANx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- ȡCANx_ACRRĴACRλ -------------*/
    tmpreg = CANx->ACRR;

    return tmpreg;
}



/**
  *   ##### (CAN)ͽպ #####
  */
/**
  *   CANģ(CAN)ͻá
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       canInitStruct: CANģϢṹָ롣
  *   ޡ
  */
void
CAN_Transmit_Message_Configuration (CAN_SFRmap* CANx,
                    CAN_MessageTypeDef* canInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_FRAME_FORMAT(canInitStruct->m_FrameFormat));
    CHECK_RESTRICTION(CHECK_CAN_REMOTE_REQUEST(canInitStruct->m_RemoteTransmit));
    CHECK_RESTRICTION(CHECK_CAN_DATA_LENGTH(canInitStruct->m_DataLength));
    CHECK_RESTRICTION(CHECK_CAN_SFF_ID(canInitStruct->m_StandardID));
    CHECK_RESTRICTION(CHECK_CAN_EFF_ID(canInitStruct->m_ExtendedID));

    if(canInitStruct->m_RemoteTransmit == CAN_REMOTE_FRAME)
    {
    	return;
    }else
    {
        /*---------------- CANx_INFRĴ ----------------*/
        /* ݽṹԱm_FrameFormatIDEλ */
        /* ݽṹԱm_RemoteTransmitRTRλ */
        /* ݽṹԱm_DataLengthDLCλ */
        tmpreg = (canInitStruct->m_FrameFormat << CAN_INFR_IDE_POS) \
               | (canInitStruct->m_RemoteTransmit << CAN_INFR_RTR_POS) \
               | (canInitStruct->m_DataLength << CAN_INFR_DLC0_POS);
        CANx->INFR = SFR_Config (CANx->INFR, ~CAN_INFR_INIT_MASK, tmpreg);

        /*----------- CANx_TX0R/CANx_TX1R/CANx_TX2RĴ -----------*/
        /* ݽṹԱm_FrameFormatѡ֡ʽ */
        /* ݽṹԱm_StandardIDIDxλ */
        /* ݽṹԱm_ExtendedIDIDxλ */
        /* ݽṹԱm_DataCANTXDTxλ */
        if (canInitStruct->m_FrameFormat != CAN_FRAME_FORMAT_EFF)
        {
            /*----------- CANx_TX0RĴ -----------*/
            tmpreg = (canInitStruct->m_Can_ID << CAN_TX0R_ID18_POS) \
                   | ((uint32_t)canInitStruct->m_Data[0] << CAN_TX0R_CANTX0_8_POS) \
                   | ((uint32_t)canInitStruct->m_Data[1] << CAN_TX0R_CANTX0_0_POS);
            CANx->TX0R = SFR_Config (CANx->TX0R, ~CAN_TX0R_SFF_MASK, tmpreg);

            /*----------- CANx_TX1RĴ -----------*/
            tmpreg = ((uint32_t)canInitStruct->m_Data[2] << CAN_TX1R_CANTX1_24_POS) \
                   | ((uint32_t)canInitStruct->m_Data[3] << CAN_TX1R_CANTX1_16_POS) \
                   | ((uint32_t)canInitStruct->m_Data[4] << CAN_TX1R_CANTX1_8_POS) \
                   | ((uint32_t)canInitStruct->m_Data[5] << CAN_TX1R_CANTX1_0_POS);
            CANx->TX1R = tmpreg;

            /*----------- CANx_TX2RĴ -----------*/
            tmpreg = ((uint32_t)canInitStruct->m_Data[6] << CAN_TX2R_CANTX2_24_POS) \
                   | ((uint32_t)canInitStruct->m_Data[7] << CAN_TX2R_CANTX2_16_POS);
            CANx->TX2R = SFR_Config (CANx->TX2R, ~CAN_TX2R_SFF_MASK, tmpreg);
        }
        else
        {
            /*----------- CANx_TX0RĴ -----------*/
            tmpreg = canInitStruct->m_Can_ID << CAN_TX0R_ID0_POS;
            CANx->TX0R = SFR_Config (CANx->TX0R, ~CAN_TX0R_EFF_MASK, tmpreg);

            /*----------- CANx_TX1RĴ -----------*/
            tmpreg = ((uint32_t)canInitStruct->m_Data[0] << CAN_TX1R_CANTX1_24_POS) \
                   | ((uint32_t)canInitStruct->m_Data[1] << CAN_TX1R_CANTX1_16_POS) \
                   | ((uint32_t)canInitStruct->m_Data[2] << CAN_TX1R_CANTX1_8_POS) \
                   | ((uint32_t)canInitStruct->m_Data[3] << CAN_TX1R_CANTX1_0_POS);
            CANx->TX1R = tmpreg;

            /*----------- CANx_TX2RĴ -----------*/
            tmpreg = ((uint32_t)canInitStruct->m_Data[4] << CAN_TX2R_CANTX2_24_POS) \
                   | ((uint32_t)canInitStruct->m_Data[5] << CAN_TX2R_CANTX2_16_POS) \
                   | ((uint32_t)canInitStruct->m_Data[6] << CAN_TX2R_CANTX2_8_POS) \
                   | ((uint32_t)canInitStruct->m_Data[7] << CAN_TX2R_CANTX2_0_POS);
            CANx->TX2R = tmpreg;
        }
    }
}


/**
  *   CANΡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Acceptance: ΣȡֵΪ32λЧֵ
  *   ޡ
  */
void
CAN_Acceptance_Mask_Config (CAN_SFRmap* CANx, uint32_t Acceptance)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- CANx_MSKRĴMSKλ -------------*/
    CANx->MSKR = Acceptance;
}

/**
  *   ȡCANΡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ΣȡֵΪ32λЧֵ
  */
uint32_t
CAN_Get_Acceptance_Mask (CAN_SFRmap* CANx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- ȡCANx_MSKRĴMSKλ -------------*/
    tmpreg = CANx->MSKR;

    return tmpreg;
}

/**
  *   ˲չʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 NewState TRUE/FALSE
  *   ޡ
  */
void
CAN_Expand_Acceptance_Enable(CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*------------- CANx_CTLRĴFILTEREXTλ -------------*/
    CANx->CTLR = SFR_Config(CANx->CTLR, ~CAN_CTLR_FILTEREXT, NewState<<CAN_CTLR_FILTEREXT_POS);
}

/**
  *   CANմչ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	  FILTER_Group CAN_Filter_G1~CAN_Filter_G8
  *       Acceptance: մ룬ȡֵΪ32λЧֵ
  *   ޡ
  */
void
CAN_Acceptance_FILTER_Config (CAN_SFRmap* CANx, uint8_t FILTER_Group, uint32_t Acceptance)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ACCEPTANCE(FILTER_Group));

    /*------------- CAN˲չ -------------*/
    *((uint32_t *)(CAN4_FILTER_ADDR+FILTER_Group*0x08)) = Acceptance;
}

/**
  *   ȡCANմչ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 FILTER_Group CAN_Filter_G1~CAN_Filter_G8
  *   մ룬ȡֵΪ32λЧֵ
  */
uint32_t
CAN_Get_Acceptance_FILTER (CAN_SFRmap* CANx, uint8_t FILTER_Group)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ACCEPTANCE(FILTER_Group));

    /*------------- ȡCAN˲չ  -------------*/
    tmpreg = *((uint32_t *)(CAN4_FILTER_ADDR+FILTER_Group*0x08));

    return tmpreg;
}

/**
  *   CANչ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	  FILTER_Group CAN_Filter_G1~CAN_Filter_G8
  *       Acceptance: ΣȡֵΪ32λЧֵ
  *   ޡ
  */
void
CAN_Acceptance_FILTER_Mask_Config (CAN_SFRmap* CANx, uint8_t FILTER_Group, uint32_t Acceptance)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ACCEPTANCE(FILTER_Group));

    /*------------- CAN˲չ -------------*/
    *((uint32_t *)(CAN4_FILTER_ADDR+FILTER_Group*0x08+0x04)) = Acceptance;
}

/**
  *   ȡCANչ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 FILTER_Group CAN_Filter_G1~CAN_Filter_G8
  *   ΣȡֵΪ32λЧֵ
  */
uint32_t
CAN_Get_Acceptance_FILTER_Mask (CAN_SFRmap* CANx, uint8_t FILTER_Group)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_ACCEPTANCE(FILTER_Group));

    /*------------- ȡCAN˲չ  -------------*/
    tmpreg = *((uint32_t *)(CAN4_FILTER_ADDR+FILTER_Group*0x08+0x04));

    return tmpreg;
}

/**
  *   RXϢ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ޡ
  */
void
CAN_Clear_CANRMC(CAN_SFRmap* CANx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*------------- CANx_CTLRĴRELRXALLλ -------------*/
    SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRXALL_POS);
    while(!((CANx->CTLR & CAN_CTLR_RELRXALL)>>CAN_CTLR_RELRXALL_POS));
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRXALL_POS);
}

/**
  *   CANʱģʽѡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 CLK_DIV  CAN_CLK_DIV2ʱ CANCKS<1:0>ѡĬ2Ƶ
  * 	 		   CAN_CLK_DIV1ʱ CANCKS<1:0>ѡĬϲƵ
  *   ޡ
  */
void
CAN_Work_Clk_Div_Select(CAN_SFRmap* CANx, uint32_t CLK_DIV)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_CLK_DIV(CLK_DIV));

    /*------------- CANx_CTLRĴCKMODEλ -------------*/
    CANx->CTLR = SFR_Config(CANx->CTLR, ~CAN_CTLR_CKMODE, CLK_DIV);
}

/**
  *   BUS OFFʱԶָʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 NewState TRUE/FALSE
  *   ޡ
  */
void
CAN_BUSS_OFF_Auto_Reset_Enable(CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*------------- CANx_CTLRĴBOFFRXEλ -------------*/
    CANx->CTLR = SFR_Config(CANx->CTLR, ~CAN_CTLR_BOFFREC, NewState<<CAN_CTLR_BOFFREC_POS);
}

/**
  *   ûػģʽѡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  * 	 LB_MODE:  CAN_Extern_LoopBack:ⲿػ
  * 	 		   CAN_Intern_LoopBack:ڲػ
  *   ޡ
  */
void
CAN_LoopBack_Mode_Select(CAN_SFRmap* CANx, uint32_t LB_MODE)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_LB_MODE(LB_MODE));

    /*------------- CANx_CTLRĴLBSELλ -------------*/
    CANx->CTLR = SFR_Config(CANx->CTLR, ~LB_MODE, LB_MODE);
}


/**
  *   ##### (CAN)ú #####
  */


/**
  *   CAN ־
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ޡ
  */
void
CAN_Clear_Buffer_Overflow_Flag (CAN_SFRmap* CANx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*-------------------- CANx_CTLRĴRELFλ --------------------*/
    SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRX_POS);
    while(!((CANx->CTLR & CAN_CTLR_RXBSTA)>>CAN_CTLR_RXBSTA_POS));
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRX_POS);

    SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_RELF_POS);
    while((CANx->CTLR & CAN_CTLR_DOSTA)>>CAN_CTLR_DOSTA_POS);
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_RELF_POS);
}

/**
  *   ͷCAN ջ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *    	ReleaseCount: ͷŻ
  *   ޡ
  */
void
CAN_Release_Receive_Buffer (CAN_SFRmap* CANx, uint32_t ReleaseCount)
{
	uint32_t i,tmprmc;
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*-------------------- CANx_CTLRĴRELRXλ --------------------*/
    for(i=0;i<ReleaseCount;i++)
    {
        tmprmc = (CANx->CTLR&CAN_CTLR_CANRMC)>>CAN_CTLR_CANRMC0_POS;
        if(tmprmc == 0)
        {
        	return ;
        }
        else
        {
    		SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRX_POS);
    		while(tmprmc - ((CANx->CTLR&CAN_CTLR_CANRMC)>>CAN_CTLR_CANRMC0_POS) != 1);
    		SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_RELRX_POS);
    	}
    }
}

/**
  *   ηͣٲöʧʱطηͣ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ޡ
  * ע֮ǰȷ߿ڿ״̬
  */
void
CAN_Transmit_Single (CAN_SFRmap* CANx)
{
	volatile unsigned int	temp;
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*-------------------- CANx_CTLRĴTXRλ --------------------*/
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_ATX_POS);
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_TXR_POS);
    CANx->CTLR = CANx->CTLR | 0x300;
    while(!(CANx->CTLR & CAN_CTLR_TXSTA));
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_ATX_POS);
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_TXR_POS);
}

/**
  *   ͣٲöʧͽʱطͣ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   ޡ
  * ע֮ǰȷ߿ڿ״̬
  */
void
CAN_Transmit_Repeat (CAN_SFRmap* CANx)
{
	volatile unsigned int	temp;
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));

    /*-------------------- CANx_CTLRĴTXRλ --------------------*/
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_ATX_POS);
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_TXR_POS);
    SFR_SET_BIT_ASM(CANx->CTLR, CAN_CTLR_TXR_POS);
    while(!(CANx->CTLR & CAN_CTLR_TXSTA));
    SFR_CLR_BIT_ASM(CANx->CTLR, CAN_CTLR_TXR_POS);

}

/**
  *   CANͻ֡ʽ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       FrameFormat: ֡ʽȡֵΪ:
  *                      CAN_FRAME_FORMAT_SFF: ׼֡ʽSFF
  *                      CAN_FRAME_FORMAT_EFF: չ֡ʽEFF
  *   ޡ
  */
void
CAN_Frame_Format_Config (CAN_SFRmap* CANx, uint32_t FrameFormat)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_FRAME_FORMAT(FrameFormat));

    /*---------------- CANx_INFRĴIDEλ ----------------*/
    if (FrameFormat != CAN_FRAME_FORMAT_SFF)
    {
        /* չ֡ʽEFF */
        SFR_SET_BIT_ASM(CANx->INFR, CAN_INFR_IDE_POS);
    }
    else
    {
        /* ׼֡ʽSFF */
        SFR_CLR_BIT_ASM(CANx->INFR, CAN_INFR_IDE_POS);
    }
}

/**
  *   CANͻԶ̷
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       RemoteRequest: Զ̷ȡֵΪ:
  *                        CAN_DATA_FRAME: ֡
  *                        CAN_REMOTE_FRAME: Զ֡
  *   ޡ
  */
void
CAN_Remote_Request_Config (CAN_SFRmap* CANx, uint32_t RemoteRequest)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_REMOTE_REQUEST(RemoteRequest));

    /*---------------- CANx_INFRĴRTRλ ----------------*/
    if (RemoteRequest != CAN_DATA_FRAME)
    {
        /* Զ֡ */
        SFR_SET_BIT_ASM(CANx->INFR, CAN_INFR_RTR_POS);
    }
    else
    {
        /* ֡ */
        SFR_CLR_BIT_ASM(CANx->INFR, CAN_INFR_RTR_POS);
    }
}

/**
  *   CANͻݳȣ8ݳȴǲõġ
  *       88ֽڼơ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Length: CANݳȣȡֵΪ4λЧֵ88ֽڼơ
  *   ޡ
  */
void
CAN_Data_Length_Config (CAN_SFRmap* CANx, uint32_t Length)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_DATA_LENGTH(Length));

    /*------------- CANx_INFRĴDLCλ -------------*/
    tmpreg = Length << CAN_INFR_DLC0_POS;
    CANx->EROR = SFR_Config (CANx->EROR, ~CAN_INFR_DLC, tmpreg);
}

/**
  *   CANͻʶ롣
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       Format: CAN֡ʽȡֵΪ:
                    CAN_FRAME_FORMAT_SFF: ׼ʽSFF
                    CAN_FRAME_FORMAT_EFF: չʽEFF
  *       IDCode: CANʶ룬׼ʽSFFΪ11λչʽEFFλ29λ
  *   ޡ
  */
void
CAN_Identification_Code_Config (CAN_SFRmap* CANx,
                    uint32_t FrameFormat, uint32_t IDCode)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_FRAME_FORMAT(FrameFormat));
    if (FrameFormat != CAN_FRAME_FORMAT_SFF)
    {
        CHECK_RESTRICTION(CHECK_CAN_EFF_ID(IDCode));
    }
    else
    {
        CHECK_RESTRICTION(CHECK_CAN_SFF_ID(IDCode));
    }

    /*------------- CANx_TX0RĴIDxλ -------------*/
    if (FrameFormat != CAN_FRAME_FORMAT_SFF)
    {
        /* չʽEFF */
        tmpreg = IDCode << CAN_TX0R_ID0_POS;
        CANx->TX0R = SFR_Config (CANx->TX0R, ~CAN_TX0R_EFF_ID, tmpreg);
    }
    else
    {
        /* ׼ʽSFF */
        tmpreg = IDCode << CAN_TX0R_ID18_POS;
        CANx->TX0R = SFR_Config (CANx->TX0R, ~CAN_TX0R_SFF_ID, tmpreg);
    }
}
/**
  *   ##### (CAN)ͽպ #####
  */


/**
  *   ##### (CAN)жϹ #####
  */
/**
  *   ȡCANжϱ־
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       InterruptType: CANжͣȡֵΧΪṩһ:
  *                        CAN_INT_RECEIVE: ж
  *                        CAN_INT_TRANSMIT: ж
  *                        CAN_INT_ERROR_ALARM: 󱨾ж
  *                        CAN_INT_DATA_OVERFLOW: ж
  *                        CAN_INT_WAKE_UP: ж
  *                        CAN_INT_ERROR_NEGATIVE: ж
  *                        CAN_INT_ARBITRATION_LOST: ٲöʧж
  *                        CAN_INT_BUS_ERROR: ߴж
  *                        CAN_INT_RECEIVED_SUCCESSFULLY: ɹж
  *                        CAN_INT_BUSS_OFF: ߹رж
  *   1:жϣ0:δжϡ
  */
FlagStatus
CAN_Get_INT_Flag (CAN_SFRmap* CANx, uint32_t InterruptType)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_INT_ONE_EVENT(InterruptType));

    /* жϱ־ */
    tmpreg = InterruptType << CAN_IFR_CANRXIF_POS;
    /*---------------- ȡCAN_IFRĴжϱ־λ ----------------*/
    if (CANx->IFR & tmpreg)
    {
        /* ж */
        return SET;
    }
    else
    {
        /* δж */
        return RESET;
    }
}

/**
  *   CANжϱ־
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       InterruptType: CANжͣȡֵΧΪṩһ:
  *                        CAN_INT_TRANSMIT: ж
  *                        CAN_INT_ERROR_ALARM: 󱨾ж
  *                        CAN_INT_DATA_OVERFLOW: ж
  *                        CAN_INT_WAKE_UP: ж
  *                        CAN_INT_ERROR_NEGATIVE: ж
  *                        CAN_INT_ARBITRATION_LOST: ٲöʧж
  *                        CAN_INT_BUS_ERROR: ߴж
  *                        CAN_INT_RECEIVED_SUCCESSFULLYɹж
  *                        CAN_INT_BUSS_OFF: ߹رж
  *   ޡ
  */
void
CAN_Clear_INT_Flag (CAN_SFRmap* CANx, uint32_t InterruptType)
{
	uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_INT_EVENT(InterruptType));

    /*---------------- CAN_IERĴжϱ־λ ----------------*/
    tmpreg = InterruptType << CAN_IFR_CANRXIF_POS;

    if(InterruptType == CAN_INT_RECEIVED_SUCCESSFULLY)
    {
    	 CANx->IER |= InterruptType << 6;
    }else
    {
    	 CANx->IER |= InterruptType << CAN_IER_CANRXIC_POS;
    }
    while((CANx->IFR & tmpreg)>0);
    CANx->IER &= ~(InterruptType << CAN_IER_CANRXIC_POS);
}

/**
  *   CANжʹܡ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *       InterruptType: CANжͣȡֵΧΪṩһ:
  *                        CAN_INT_RECEIVE: ж
  *                        CAN_INT_TRANSMIT: ж
  *                        CAN_INT_ERROR_ALARM: 󱨾ж
  *                        CAN_INT_DATA_OVERFLOW: ж
  *                        CAN_INT_WAKE_UP: ж
  *                        CAN_INT_ERROR_NEGATIVE: ж
  *                        CAN_INT_ARBITRATION_LOST: ٲöʧж
  *                        CAN_INT_BUS_ERROR: ߴж
  *                        CAN_INT_RECEIVED_SUCCESSFULLY: ɹж
  *                        CAN_INT_BUSS_OFF: ߹رж
  *       NewState: DMAͨжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
CAN_Set_INT_Enable (CAN_SFRmap* CANx,
                    uint32_t InterruptType, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_CAN_INT_EVENT(InterruptType));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- CAN_IERĴжʹλ ----------------*/
	if (NewState != FALSE)
	{
		/* ʹж */
		CANx->IER |= InterruptType << CAN_IER_CANRXIE_POS;
	}
	else
	{
		/* ֹж */
		CANx->IER &= ~(InterruptType << CAN_IER_CANRXIE_POS);
	}
}

/**
  *   CANDMAʹ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *      NewState: TRUE/FALSE
  *   ޡ
  */
void
CAN_Set_Reseive_DMA_Enable(CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    CANx->IER = SFR_Config(CANx->IER, ~CAN_IER_CANRXDE, NewState<<CAN_IER_CANRXDE_POS);
}

/**
  *   CANDMAʹ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *      NewState: TRUE/FALSE
  *   ޡ
  */
void
CAN_Set_Send_DMA_Enable(CAN_SFRmap* CANx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    CANx->IER = SFR_Config(CANx->IER, ~CAN_IER_CANTXDE, NewState<<CAN_IER_CANTXDE_POS);
}

/**
  *   ȡCAN DMA־λ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   1:0:δ
  */
FlagStatus
CAN_Get_Receive_DMA_Flag(CAN_SFRmap* CANx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    if (CANx->IFR & CAN_IFR_CANRXDE)
    {
        /*  */
        return SET;
    }
    else
    {
        /* δ */
        return RESET;
    }
}

/**
  *   ȡCAN DMA־λ
  *   CANx: ָCANڴṹָ룬ȡֵΪCAN4_SFR
  *   1:0:δ
  */
FlagStatus
CAN_Get_Send_DMA_Flag(CAN_SFRmap* CANx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_CAN_ALL_PERIPH(CANx));
    if (CANx->IFR & CAN_IFR_CANTXDE)
    {
        /*  */
        return SET;
    }
    else
    {
        /* δ */
        return RESET;
    }
}

/**
  *   ##### (CAN)жϹ #####
  */


